/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.utils.conversation;

import java.lang.reflect.Method;
import java.util.List;

import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.tmcdb.service.AcaCorrDelaysService;
import alma.obops.dam.utils.ConversationInterceptor;
import alma.obops.dam.utils.ConversationTokenProvider;
import alma.obops.dam.utils.ConversationTokenProviderAdapter;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.tmcdb.domain.AcaCorrDelays;
import alma.tmcdb.history.HistoryRecord;

/**
 * @author sharring
 *
 */
public class AcaCorrDelaysConversationUtils 
{
	private static AcaCorrDelaysConversationUtils singletonInstance;

	private AcaCorrDelaysConversationUtils() 
	{
	}

	public static synchronized AcaCorrDelaysConversationUtils getInstance()
	{
		if(null == singletonInstance)
		{
			singletonInstance = new AcaCorrDelaysConversationUtils();
		}

		return singletonInstance;
	}

	/**
	 * @param acaCorrDel
	 */
	public void endAcaCorrDelaysSave(AcaCorrDelays acaCorrDel) throws Exception
	{
		Method methodToInvoke = AcaCorrDelaysConversationUtils.class.getMethod("privateEndAcaCorrDelaysSave",
				AcaCorrDelays.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = acaCorrDel;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateEndAcaCorrDelaysSave(AcaCorrDelays acaCorrDel) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AcaCorrDelaysService service = TmcdbContextFactory.INSTANCE.getAcaCorrDelaysService();
		service.endSave(acaCorrDel);
		return retVal;
	}

	/**
	 * @param acaCorrDelays the AcaCorrDelays which we wish to save
	 * @param userId the userid of the person making the change
	 * @param description a description of the change
	 * @return a boolean indicating the AcaCorrDelays can be saved (true) or not (false)
	 * @throws exception if there's a problem
	 */
	public boolean prepareAcaCorrDelaysSave(AcaCorrDelays acaCorrDelays, String userId, String description) throws Exception 
	{
		Method methodToInvoke = AcaCorrDelaysConversationUtils.class.getMethod("privatePrepareAcaCorrDelaysSave", 
				AcaCorrDelays.class, String.class, String.class, BooleanHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		BooleanHolder resultholder = new BooleanHolder();
		Object[] args = new Object[4];
		args[0] = acaCorrDelays;
		args[1] = userId;
		args[2] = description;
		args[3] = resultholder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		boolean retVal = resultholder.getBooleanValue();
		return retVal;
	}
	
	public ConversationTokenProvider privatePrepareAcaCorrDelaysSave(AcaCorrDelays acaCorrDel, String who, String description, BooleanHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AcaCorrDelaysService service = TmcdbContextFactory.INSTANCE.getAcaCorrDelaysService();
		boolean successful = service.prepareSave(acaCorrDel, who, description);
		resultHolder.setBooleanValue(successful);
		return retVal;
	}

	/**
	 * @param acaCorrDelays
	 */
	public void saveOrUpdateAcaCorrDelays(AcaCorrDelays acaCorrDel) throws Exception
	{
		Method methodToInvoke = AcaCorrDelaysConversationUtils.class.getMethod("privateSaveOrUpdateAcaCorrDelays",
				AcaCorrDelays.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		Object[] args = new Object[1];
		args[0] = acaCorrDel;
		conversationInterceptor.invoke(methodToInvoke, this, args);
	}
	
	public ConversationTokenProvider privateSaveOrUpdateAcaCorrDelays(AcaCorrDelays acaCorrDel) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AcaCorrDelaysService service = TmcdbContextFactory.INSTANCE.getAcaCorrDelaysService();
		service.update(acaCorrDel);
		return retVal;
	}

	/**
	 * @param acaCorrDelays
	 * @param referenceRecord
	 * @return
	 */
	public AcaCorrDelays getHistoricalAcaCorrDelays(AcaCorrDelays acaCorrDelays, HistoryRecord clickedRecord) throws Exception 
	{
		Method methodToInvoke = AcaCorrDelaysConversationUtils.class.getMethod("privateGetHistoricalAcaCorrDelays", AcaCorrDelays.class, HistoryRecord.class, AcaCorrDelaysHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		AcaCorrDelaysHolder holder = new AcaCorrDelaysHolder();
		Object[] args = new Object[3];
		args[0] = acaCorrDelays;
		args[1] = clickedRecord;
		args[2] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		AcaCorrDelays retVal = holder.getAcaCorrDelays();
		return retVal;
	}
	
	public ConversationTokenProvider privateGetHistoricalAcaCorrDelays(AcaCorrDelays acaCorrDel, HistoryRecord record, AcaCorrDelaysHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AcaCorrDelaysService service = TmcdbContextFactory.INSTANCE.getAcaCorrDelaysService();
		AcaCorrDelays historicalAcaCorrDelays = service.getHistoricalAcaCorrDelays(acaCorrDel, record.getVersion());
		resultHolder.setAcaCorrDelays(historicalAcaCorrDelays);
		return retVal;
	}


	/**
	 * @param acaCorrDel
	 * @return a list of historyrecord's for the acacorrdelays object
	 * @throws Exception 
	 */
	public List<HistoryRecord> getAcaCorrDelaysHistory(AcaCorrDelays acaCorrDel) throws Exception {
		Method methodToInvoke = AcaCorrDelaysConversationUtils.class.getMethod("privateGetAcaCorrDelaysHistory", AcaCorrDelays.class, HistoryRecordListHolder.class);	
		ConversationInterceptor conversationInterceptor = TmcdbContextFactory.INSTANCE.getConversationInterceptor();
		HistoryRecordListHolder holder = new HistoryRecordListHolder();
		Object[] args = new Object[2];
		args[0] = acaCorrDel;
		args[1] = holder;
		conversationInterceptor.invoke(methodToInvoke, this, args);
		return holder.getHistoryRecords();

	}

	public ConversationTokenProvider privateGetAcaCorrDelaysHistory(AcaCorrDelays acaCorrDel, HistoryRecordListHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		AcaCorrDelaysService service = TmcdbContextFactory.INSTANCE.getAcaCorrDelaysService();
		List<HistoryRecord> results = service.getHistory(acaCorrDel);
		resultHolder.setHistoryRecords(results);
		return retVal;
	}
	
	private class AcaCorrDelaysHolder
	{
		private AcaCorrDelays acaCorrDelays;
		
		public AcaCorrDelays getAcaCorrDelays() {
			return acaCorrDelays;
		}
		
		public void setAcaCorrDelays(AcaCorrDelays acaCorrDels) {
			this.acaCorrDelays = acaCorrDels;
		}
	}
}
