/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * LabelHelper.java
 */
package alma.obops.tmcdbgui.utils;

import alma.acs.tmcdb.AcsService;
import alma.acs.tmcdb.AcsServiceServiceType;
import alma.acs.tmcdb.BACIProperty;
import alma.acs.tmcdb.ChannelMapping;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.Computer;
import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.DomainsMapping;
import alma.acs.tmcdb.NotificationServiceMapping;
import alma.tmcdb.domain.Assembly;
import alma.tmcdb.domain.HwConfiguration;

/**
 * The <code>LabelHelper</code> class is a utility class for helping construct
 * several labels across the TMCDB Explorer. The need of this class arises from
 * the fact that there are some objects whose "detailed" labels are complex to construct
 * (e.g., if we want to see the full path and name for a component, a concatenation must
 * be done, including trims and search/replace). Therefore, this class exposes static
 * methods that deal with these issues.
 * 
 * @author rtobar, Mar 9, 2010
 *
 */
public class LabelHelper {

	/**
	 * Computes the full path for a given <code>Component</code> object.
	 * @param c The component
	 * @param initialSlash Whether the path should include a initial <code>/</code> or not
	 * @return The full path for the component
	 */
	public static String getFullPath(Component c, boolean initialSlash) {

		StringBuilder sb = new StringBuilder();
		sb.append("/")
		  .append( c.getPath() != null ? c.getPath(): "" )
		  .append("/")
		  .append(c.getComponentName());

		String s = sb.toString().replaceAll("/+", "/");
		if( !initialSlash )
			s = s.replaceAll("^/", "");
		return s;
	}
	
	/**
	 * Computes the full path for a given <code>BACIProperty</code> object.
	 * @param c The BACIProperty
	 * @param initialSlash Whether the path should include a initial <code>/</code> or not
	 * @return The full path for the component
	 */
	public static String getFullPath(BACIProperty baciProp, boolean initialSlash) {
		String retVal = baciProp.getPropertyName() != null ? baciProp.getPropertyName(): "";
		return retVal;
	}


	/**
	 * Computes the full path for a given <code>Container</code> object.
	 * @param c The container
	 * @param initialSlash Whether the path should include a initial <code>/</code> or not
	 * @return The full path for the container
	 */
	public static String getFullPath(Container c, boolean initialSlash) {

		StringBuilder sb = new StringBuilder();
		sb.append("/")
		  .append( c.getPath() != null ? c.getPath(): "" )
		  .append("/")
		  .append(c.getContainerName());

		String s = sb.toString().replaceAll("/+", "/");
		if( !initialSlash )
			s = s.replaceAll("^/", "");
		return s;
	}

	/**
	 * Computes a nice label for a given computer: Name (hostname)
	 * @param c The computer
	 * @return The nice label
	 */
	public static String getComputerLabel(Computer c) {
		return (c.getName() != null ? c.getName() : "<unnamed>" ) + " (" + c.getNetworkName() + ")";
	}
	
	public static String getConfigurationLabel(HwConfiguration config)
	{
		return (config.getName() != null ? config.getName() : "<unnamed>");
	}

	public static String getAssemblyLabel(Assembly assembly) 
	{
		StringBuffer retBuffer = new StringBuffer();
		retBuffer.append(assembly.getAssemblyType().getName());
		retBuffer.append(":").append(assembly.getSerialNumber() != null ? assembly.getSerialNumber() : "<unnamed>");
		
		return retBuffer.toString();
	}
	
	public static String getAcsServiceLabel(AcsService service) 
	{
		StringBuffer buffer = new StringBuffer();
		buffer.append(service.getServiceType().toString());
		if(service.getServiceType().equals(AcsServiceServiceType.NOTIFICATION)) {
			buffer.append(".").append(service.getServiceInstanceName());
		}
		return buffer.toString();
	}

	public static String getDomainsMappingLabel(DomainsMapping domainsMapping) {
		StringBuffer buffer = new StringBuffer();
		buffer.append("Domain mapping: " + domainsMapping.getName());
		return buffer.toString();
	}

	public static String getChannelMappingLabel(ChannelMapping channelMapping) {
		StringBuffer buffer = new StringBuffer();
		String name = channelMapping.getName() != null ? channelMapping.getName() : "<new>";
		buffer.append("Channel mapping: " + name);
		return buffer.toString();
	}

	public static String getNotificationServiceMappingLabel(NotificationServiceMapping nsMapping) {
		StringBuffer buffer = new StringBuffer();
		buffer.append("Notification Service mapping: " + nsMapping.getDefaultNotificationService());
		return buffer.toString();
	}
}
