/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * GuiUtils.java
 *
 * Copyright European Southern Observatory 2008
 */

package alma.obops.tmcdbgui.utils;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Drawable;
import org.eclipse.swt.graphics.FontMetrics;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

/**
 * A collection of GUI utility methods and constants
 *
 * @author amchavan, Oct 8, 2008
 * 
 */



public class GuiUtils 
{
	private static final String SECRET_DEBUG_FLAG = "GOD_USER_TMCDB_OVERRIDE";
	private static final String ALMAMGR_USER_ID = "almamgr";
	private static final String USER_NAME_PROPERTY = "user.name";

	//-------------------------------------------------------
    // The following constants must all have unique values!
    // (cannot use an enumeration)
    //-------------------------------------------------------
	/** Changed property name: an Antenna has changed */
    public static final int CHANGE_ANTENNA = 1;

    /** Changed property name: a StartupScenario has changed */
    public static final int CHANGE_STARTUP_SCENARIO = 2;

    /** Changed property name: an Antenna was dropped onto a StartupScenario */
    public static final int DROP_ANTENNA = 3;

    /** Changed property name: a FrontEnd was dropped onto a StartupScenario */
    public static final int DROP_FRONT_END = 4;

    /** Changed property name: a CentralRack was dropped onto a StartupScenario */
    public static final int DROP_CENTRAL_RACK = 5;

    /** Changed property name: a MasterClock was dropped onto a StartupScenario */
    public static final int DROP_MASTER_CLOCK = 6;
    
    /** Changed property name: a PhotonicReference was dropped onto a StartupScenario */
    public static final int DROP_PHOTONIC_REFERENCE = 7;
    
    /** Changed property name: a WeatherStation was dropped onto a StartupScenario */
    public static final int DROP_WEATHER_STATION = 8;

    /** Changed property name: a pad has changed */
	public static final int CHANGE_PAD = 0;

	public static final String LOG_LEVEL_NOT_SPECIFIED = "<Not specified>";

	private static final String[] logLevelStrings = {
		LOG_LEVEL_NOT_SPECIFIED,
		alma.AcsLogLevels.TRACE_NAME.value,
		alma.AcsLogLevels.DEBUG_NAME.value,
		alma.AcsLogLevels.INFO_NAME.value,
		alma.AcsLogLevels.NOTICE_NAME.value,
		alma.AcsLogLevels.WARNING_NAME.value,
		alma.AcsLogLevels.ERROR_NAME.value,
		alma.AcsLogLevels.CRITICAL_NAME.value,
		alma.AcsLogLevels.ALERT_NAME.value,
		alma.AcsLogLevels.EMERGENCY_NAME.value,
		alma.AcsLogLevels.OFF_NAME.value,
	};

    /**
     * Show an error dialog box
     *
     * @param parentShell
     * @param dialogTitle
     * @param dialogMessage
     */
    public static void showErrorDialog( Shell parentShell,
                                        String dialogTitle,
                                        String dialogMessage ) {
        String[] buttons = new String[] { "OK" };
        MessageDialog dialog = new MessageDialog( parentShell, dialogTitle,
                                                  null, dialogMessage,
                                                  MessageDialog.ERROR,
                                                  buttons, 0 );
        dialog.open();
    }

    /**
     * Show an informational dialog box
     *
     * @param parentShell
     * @param dialogTitle
     * @param dialogMessage
     */
    public static void showInfoDialog( Shell parentShell,
                                       String dialogTitle,
                                       String dialogMessage ) {
        String[] buttons = new String[] { "OK" };
        MessageDialog dialog = new MessageDialog( parentShell, dialogTitle,
                                                  null, dialogMessage,
                                                  MessageDialog.INFORMATION,
                                                  buttons, 0 );
        dialog.open();
    }

    /**
     * @param parent
     * @param style
     * @return
     */
    public static Combo createLogLevelCombo( Composite parent) {
    	Combo combo = new Combo(parent, SWT.DROP_DOWN | SWT.READ_ONLY );
    	combo.setItems(logLevelStrings);
    	combo.setData("type", "logLevel");
    	return combo;
    }

    /**
     * This method is used to determine if the tmcdbexplorer gui should allow full-editing capabilities ("God" mode) 
     * or whether it should run in a limited-functionality mode ("Scientist" mode). The logic is currently very simple:
     * you are either logged in as almamgr, and therefore you are a "God" user; else (if you are logged in as *any*
     * other user), you are a scientist user.
     * 
     * @return boolean indicating if the user has all capabilities ("god" mode) or not.
     */
	public static boolean isGodUser() 
	{
		boolean retVal = false;
		if(System.getenv(SECRET_DEBUG_FLAG) != null) {
			retVal = true;
		}
		else {
			retVal = ALMAMGR_USER_ID.equals(System.getProperty(USER_NAME_PROPERTY));
		}
		return retVal;
	}

	public static GridData getGridDataForCharWidth(int columns, Text text) 
	{
	    GC gc = new GC(text);
	    FontMetrics fm = gc.getFontMetrics();
	    int width = columns * fm.getAverageCharWidth();
	    int height = fm.getHeight();
	    gc.dispose();
	    GridData gd = new GridData();
	    gd.widthHint = text.computeSize(width, height).x;
	    gd.minimumWidth = gd.widthHint;
	    return gd;
	}
	
	public static int getWidthForChars(int numChars, Drawable drawable)
	{
		GC gc = new GC(drawable);
		FontMetrics fm = gc.getFontMetrics();
		int width = numChars * fm.getAverageCharWidth();
		gc.dispose();
		return width;
	}

	public static boolean onlyItemsOfClassSelected(IStructuredSelection sselection, Class<?> clazz) 
	{
		boolean retVal = true;

		for(Object obj : sselection.toList())
		{
			if(!clazz.isInstance(obj)) {
				retVal = false;
			}
		}

		return retVal;
	}
}
