/*
 * ALMA - Atacama Large Millimiter Array (c) European Southern Observatory, 2011
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package alma.obops.tmcdbgui.ui;

import java.beans.BeanInfo;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;

import org.eclipse.jface.viewers.IDecoration;
import org.eclipse.jface.viewers.ILightweightLabelDecorator;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Display;
import org.hibernate.metadata.ClassMetadata;
import org.hibernate.type.IntegerType;

import alma.acs.tmcdb.translator.TmcdbObject;
import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.tmcdbgui.views.support.TmcdbObjectWrapper;

/**
 * A label decorator to be used in those views that wish to display the ID of the
 * different TMCDB objects that it displays
 *
 * @author  rtobar, Jul 22, 2011
 * @version $Id: TmcdbObjectIDLabelDecorator.java,v 1.3 2012/04/18 09:47:08 rtobar Exp $
 */
public class TmcdbObjectIDLabelDecorator extends LabelProvider implements
		ILightweightLabelDecorator {

	public static final String ID = "alma.obops.tmcdb.explorer.tmcdbObjectIdDecorator";

	@Override
	public void decorate(Object element, final IDecoration decoration) {

		Integer id = null;

		if( element instanceof TmcdbObjectWrapper )
			element = ((TmcdbObjectWrapper) element).getWrappedObject();

		// Only TmcdbObjects are decorated
		if( element instanceof TmcdbObject ) {

			// See if the identifier for this class is an integer
			// If so, then we get its reader method and read its value,
			// which should be already in memory, so it won't hit the database
			Class<? extends Object> clazz = element.getClass();
			ClassMetadata md = TmcdbContextFactory.INSTANCE.getSessionFactory().getClassMetadata(clazz);

			if( md != null && md.hasIdentifierProperty() && (md.getIdentifierType() instanceof IntegerType) ) {

				try {
					String prop = md.getIdentifierPropertyName();
					BeanInfo beanInfo = java.beans.Introspector.getBeanInfo(clazz);
					PropertyDescriptor[] descriptors = beanInfo.getPropertyDescriptors();
					for (int i = 0; i < descriptors.length; i++) {

						PropertyDescriptor propertyDescriptor = descriptors[i];
						if( !propertyDescriptor.getName().equals(prop) )
							continue;

						Method method = propertyDescriptor.getReadMethod();
						id = (Integer)method.invoke(element);
						break;
					}
				} catch(Exception e) {
					// not big deal, we just don't display the ID
					e.printStackTrace();
				}
			}
		}

		if( id != null ) {
			final Display d = Display.getDefault();
			d.syncExec(new Runnable() {
				@Override
				public void run() {
					decoration.setForegroundColor(Display.getCurrent().getSystemColor(SWT.COLOR_GRAY));
				}
			});
			decoration.addSuffix("   [" + id.toString() + "]");
		}

	}

}
