/*
 * ALMA - Atacama Large Millimiter Array (c) European Southern Observatory, 2011
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package alma.obops.tmcdbgui.ui;

import java.io.File;
import java.util.HashSet;
import java.util.Set;

import org.eclipse.jface.viewers.IDecoration;
import org.eclipse.jface.viewers.ILightweightLabelDecorator;
import org.eclipse.jface.viewers.LabelProvider;

import alma.acs.tmcdb.Component;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;

/**
 * A label decorator taht overlays a warning icon in the Component icon
 * if its library ("code" field) does not exist on the filesystem. It currently
 * works only for C++ components.
 *
 * @author  rtobar, Apr 12, 2012
 */
public class ComponentCppPyCodeLabelDecorator extends LabelProvider implements
		ILightweightLabelDecorator {

	public static final String ID = "alma.obops.tmcdb.explorer.componentCppPyCodeLabelDecorator";

	private static final String[] INSTROOTS;
	static {
		Set<String> dirs = new HashSet<String>();

		// INTROOT
		String introot = System.getenv("INTROOT");
		if( introot != null && new File(introot).exists() )
			dirs.add(introot);

		// INTLIST
		String intlist = System.getenv("INTLIST");
		if( intlist != null )
			for(String intlistMember: intlist.split(":"))
				if( new File(intlistMember).exists() )
					dirs.add(intlistMember);

		// ACSROOT
		String acsroot = System.getenv("ACSROOT");
		if( acsroot != null && new File(acsroot).exists() )
			dirs.add(acsroot);

		INSTROOTS = dirs.toArray(new String[dirs.size()]);
	}
	
	@Override
	public void decorate(Object element, final IDecoration decoration) {

		if( element == null )
			return;

		if( element instanceof Component ) {

			Component c = (Component) element;
			if( c.getCode() == null || c.getImplLang() == null )
				return;

			boolean found = false;

			switch(c.getImplLang()) {

				// In C++ we look for the corresponding lib<name>.so file
				case CPP:
					found = fileIsInstalled("lib" + File.separatorChar + "lib" + c.getCode() + ".so");
					break;

				// In Python we look for the corresponding .py file
				case PY:

					// The last part of the code is a filename,
					// the rest is a hierarchy of directories
					String[] nameParts = c.getCode().split("\\.");
					StringBuilder pyFileLocation = new StringBuilder();
					pyFileLocation.append("lib");
					pyFileLocation.append(File.separatorChar);
					pyFileLocation.append("python");
					pyFileLocation.append(File.separatorChar);
					pyFileLocation.append("site-packages");
					pyFileLocation.append(File.separatorChar);
					for(int i=0; i!=nameParts.length; i++) {
						pyFileLocation.append(nameParts[i]);
						if( i != nameParts.length - 1)
							pyFileLocation.append(File.separatorChar);
						else
							pyFileLocation.append(".py");
					}

					found = fileIsInstalled(pyFileLocation.toString());
					break;

				// In Java, we don't look for anything, it's too costly
				case JAVA:
				default:
					found = true;
					break;
			} 

			if( !found )
				decoration.addOverlay(RcpUtils.getImageDescriptor("icons/question-overlay.png"), IDecoration.BOTTOM_RIGHT);
		}

	}

	private static boolean fileIsInstalled(String fileLocation) {
		for(String instroot: INSTROOTS)
			if( new File(instroot + File.separatorChar + fileLocation).exists() )
				return true;
		return false;
	}

}