/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.search;

//import static alma.obops.tmcdbgui.search.TmcdbQueryTables.ASSEMBLY;
//import static alma.obops.tmcdbgui.search.TmcdbQueryTables.ASSEMBLY_TYPE;
import static alma.obops.tmcdbgui.search.TmcdbQueryTables.BACI_PROPERTY;
import static alma.obops.tmcdbgui.search.TmcdbQueryTables.COMPONENT;
import static alma.obops.tmcdbgui.search.TmcdbQueryTables.COMPONENT_TYPE;
import static alma.obops.tmcdbgui.search.TmcdbQueryTables.COMPUTER;
import static alma.obops.tmcdbgui.search.TmcdbQueryTables.CONFIGURATION;
import static alma.obops.tmcdbgui.search.TmcdbQueryTables.CONTAINER;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Status;
import org.eclipse.search.ui.ISearchQuery;
import org.eclipse.search.ui.ISearchResult;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Restrictions;

import alma.obops.tmcdbgui.utils.conversation.BACIPropertyConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.ComponentConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.ComponentTypeConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.ComputerConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.ContainerConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.SwConfigurationConversationUtils;

public class TmcdbSearchQuery implements ISearchQuery {

	private String searchString;
	private boolean isCaseInsensitive;
	private boolean isWholeString;
	private List<TmcdbQueryTables> tablesToQuery;
	private TmcdbSearchResult searchResult;

	public TmcdbSearchQuery(String searchString, boolean isCaseInsensitive, boolean isWholeString, List<TmcdbQueryTables> tablesToQuery) {
		this.searchString = searchString;
		this.isCaseInsensitive = isCaseInsensitive;
		this.tablesToQuery = tablesToQuery;
		this.isWholeString = isWholeString;
		this.searchResult = new TmcdbSearchResult(this);
	}

	@Override
	public IStatus run(IProgressMonitor monitor) throws OperationCanceledException {

		monitor.beginTask("Searching for objects", tablesToQuery.size());
		searchResult.clear();

		// For each table, we look all possible matches, and add them to the result
		for(TmcdbQueryTables table: tablesToQuery) {
			monitor.subTask("Searching for " + table.getNiceName());

			switch(table) {
				case ASSEMBLY:
					searchAssemblies();
					break;
				case ASSEMBLY_TYPE:
					searchAssemblyTypes();
					break;
				case BACI_PROPERTY:
					searchBACIProperties();
					break;
				case COMPONENT:
					searchComponents();
					break;
				case COMPONENT_TYPE:
					searchComponentTypes();
					break;
				case COMPUTER:
					searchComputers();
					break;
				case CONFIGURATION:
					searchConfigurations();
					break;
				case CONTAINER:
					searchContainers();
					break;
			}

			monitor.worked(1);
		}
		monitor.done();
		searchResult.publishResults();

		return Status.OK_STATUS;
	}

	@Override
	public String getLabel() {
		return "'" + (isCaseInsensitive ? "%" : "") + searchString + (isCaseInsensitive ? "%" : "") + "'";
	}

	@Override
	public boolean canRerun() {
		return true;
	}

	@Override
	public boolean canRunInBackground() {
		return true;
	}

	@Override
	public ISearchResult getSearchResult() {
		return searchResult;
	}

	/* ==================================== */
	/* | SEARCHER METHODS                 | */
	/* ==================================== */
	private void searchAssemblies() {
//		List<Object> criteria = createCriteriaForProperties("serialNumber");
//		List<?> result = AssemblyConversationUtils.getInstance().find(criteria, null);
//		for(Object resultDetail: result) {
//			searchResult.addElement(ASSEMBLY, resultDetail);
//		}
	}

	private void searchAssemblyTypes() {
//		List<Object> criteria = createCriteriaForProperties("name", "fullName");
//		List<?> result = AssemblyTypeConversationUtils.getInstance().find(criteria, null);
//		for(Object resultDetail: result) {
//			searchResult.addElement(ASSEMBLY_TYPE, resultDetail);
//		}
	}

	private void searchBACIProperties() {
		List<Object> criteria = createCriteriaForProperties("propertyName", "description");
		List<?> result = null;
		try {
			result = BACIPropertyConversationUtils.getInstance().find(criteria, null);
		}
		catch(Exception ex) {
			ex.printStackTrace();
		}
		if(result != null) {
			for(Object resultDetail: result) {
				searchResult.addElement(BACI_PROPERTY, resultDetail);
			}
		}
	}

	private void searchComponents() {
		List<Object> criteria = createCriteriaForProperties("componentName", "path");
		List<?> result = null;
		try{
			result = ComponentConversationUtils.getInstance().find(criteria, null);
		}
		catch(Exception ex) {
			ex.printStackTrace();
		}
		if(result != null) {
			for(Object resultDetail: result) {
				searchResult.addElement(COMPONENT, resultDetail);
			}
		}
	}

	private void searchComponentTypes() {
		List<Object> criteria = createCriteriaForProperties("IDL");
		List<?> result = null;
		try{
			result = ComponentTypeConversationUtils.getInstance().find(criteria, null);
		}
		catch(Exception ex) {
			ex.printStackTrace();
		}
		if(result != null) {
			for(Object resultDetail: result) {
				searchResult.addElement(COMPONENT_TYPE, resultDetail);
			}
		}
	}

	private void searchComputers() {
		List<Object> criteria = createCriteriaForProperties("networkName", "name", "physicalLocation");
		List<?> result = null;
		try{
			result = ComputerConversationUtils.getInstance().find(criteria, null);
		}
		catch(Exception ex) {
			ex.printStackTrace();
		}
		if(result != null) {
			for(Object resultDetail: result) {
				searchResult.addElement(COMPUTER, resultDetail);
			}
		}
	}

	private void searchConfigurations() {
		List<Object> criteria = createCriteriaForProperties("configurationName", "fullName");
		List<?> result = null;
		try{
			result = SwConfigurationConversationUtils.getInstance().find(criteria, null);
		}
		catch(Exception ex) {
			ex.printStackTrace();
		}
		if(result != null) {
			for(Object resultDetail: result) {
				searchResult.addElement(CONFIGURATION, resultDetail);
			}
		}
	}

	private void searchContainers() {
		List<Object> criteria = createCriteriaForProperties("containerName", "path");
		List<?> result = null;
		try{
			result = ContainerConversationUtils.getInstance().find(criteria, null);
		}
		catch(Exception ex) {
			ex.printStackTrace();
		}
		if(result != null) {
			for(Object resultDetail: result) {
				searchResult.addElement(CONTAINER, resultDetail);
			}
		}
	}

	private List<Object> createCriteriaForProperties(String ... properties) {
		List<Object> criteria = new ArrayList<Object>();

		Criterion finalCriterion = null;
		for(String property: properties) {
			Criterion c = getCriterion(property);
			if ( finalCriterion == null)
				finalCriterion = c;
			else
				finalCriterion = Restrictions.or(finalCriterion, c);
		}

		criteria.add(finalCriterion);
		return criteria;
	}

	private Criterion getCriterion(String propertyName) {

		if( isWholeString ) 
		{
			if( isCaseInsensitive ) {
				return Restrictions.ilike(propertyName, searchString, MatchMode.EXACT);
			}
			
			return Restrictions.eq(propertyName, searchString);
		}
		
		if( isCaseInsensitive ) {
			return Restrictions.ilike(propertyName, searchString, MatchMode.ANYWHERE);
		}
		
		return Restrictions.like(propertyName, searchString, MatchMode.ANYWHERE);
	}

}