/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.search;

import static alma.obops.tmcdbgui.search.TmcdbQueryTables.ASSEMBLY;
import static alma.obops.tmcdbgui.search.TmcdbQueryTables.ASSEMBLY_TYPE;
import static alma.obops.tmcdbgui.search.TmcdbQueryTables.BACI_PROPERTY;
import static alma.obops.tmcdbgui.search.TmcdbQueryTables.COMPONENT;
import static alma.obops.tmcdbgui.search.TmcdbQueryTables.COMPONENT_TYPE;
import static alma.obops.tmcdbgui.search.TmcdbQueryTables.COMPUTER;
import static alma.obops.tmcdbgui.search.TmcdbQueryTables.CONFIGURATION;
import static alma.obops.tmcdbgui.search.TmcdbQueryTables.CONTAINER;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.jface.dialogs.DialogPage;
import org.eclipse.search.ui.ISearchPage;
import org.eclipse.search.ui.ISearchPageContainer;
import org.eclipse.search.ui.NewSearchUI;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;

public class TmcdbObjectSearchPage extends DialogPage implements ISearchPage {

	private Text searchText;
	private Button caseSensitiveCheck;
	private Button wholeWord;
	private Map<TmcdbQueryTables, Button> checkButtons = new HashMap<TmcdbQueryTables, Button>();

	@Override
	public void createControl(Composite parent) {

		parent.setLayout(new GridLayout(1, true));

		GridData gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		Label label = new Label(parent, SWT.NONE);
		label.setLayoutData(gd);
		label.setText("Text to search:");

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		searchText = new Text(parent, SWT.BORDER);
		searchText.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		Group checkGroup = new Group(parent, SWT.SHADOW_NONE);
		checkGroup.setLayout(new GridLayout(2, true));
		checkGroup.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		caseSensitiveCheck = new Button(checkGroup, SWT.CHECK);
		caseSensitiveCheck.setText("case-sensitive");
		caseSensitiveCheck.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, true, false);
		wholeWord = new Button(checkGroup, SWT.CHECK);
		wholeWord.setText("exact match");
		wholeWord.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.TOP, true, true);
		gd.horizontalSpan = 2;
		Group objectTypesGroup = new Group(parent, SWT.BORDER);
		objectTypesGroup.setText("Search for...");
		objectTypesGroup.setLayoutData(gd);
		objectTypesGroup.setLayout(new GridLayout(3, true));

		addObjectType(objectTypesGroup, CONFIGURATION , true);
		addObjectType(objectTypesGroup, COMPONENT_TYPE, true);
		addObjectType(objectTypesGroup, ASSEMBLY_TYPE, true);

		addObjectType(objectTypesGroup, COMPUTER, true);
		addObjectType(objectTypesGroup, CONTAINER, true);
		addObjectType(objectTypesGroup, COMPONENT, true);

		addObjectType(objectTypesGroup, BACI_PROPERTY, true);
		addObjectType(objectTypesGroup, ASSEMBLY, true);

		// case-insensitive, match substring by default
		caseSensitiveCheck.setSelection(false);

		setControl(parent);

	}

	private void addObjectType(Group group, TmcdbQueryTables table, boolean activated) {
		GridData gd = new GridData(SWT.LEFT, SWT.CENTER, true, false);
		Button checkButton = new Button(group, SWT.CHECK);
		checkButton.setImage(RcpUtils.getImage(table.getIconName()));
		checkButton.setText(table.getNiceName());
		checkButton.setSelection(activated);
		checkButton.setLayoutData(gd);
		checkButtons.put(table, checkButton);
	}

	@Override
	public Image getImage() {
		return RcpUtils.getImage("icons/find.png");
	}

	@Override
	public String getTitle() {
		return "TMCDB objects";
	}

	@Override
	public boolean performAction() {

		// Don't search anything
		if( searchText.getText() == null || searchText.getText().trim().length() == 0 )
			return false;

		// Check which tables to search for; if none, then don't search anything
		List<TmcdbQueryTables> tablesToQuery = new ArrayList<TmcdbQueryTables>();
		for(TmcdbQueryTables table: checkButtons.keySet()) {
			if( checkButtons.get(table).getSelection() )
				tablesToQuery.add(table);
		}
		if( tablesToQuery.size() == 0 )
			return false;

		TmcdbSearchQuery query = new TmcdbSearchQuery(searchText.getText().trim(),
		       !caseSensitiveCheck.getSelection(), wholeWord.getSelection(), tablesToQuery);
		try {
			NewSearchUI.runQueryInBackground(query);
		} catch(IllegalArgumentException ex) {
			ex.printStackTrace();
			return false;
		}

		return true;
	}

	@Override
	public void setContainer(ISearchPageContainer container) { }

}
