/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.search;

import java.util.List;
import java.util.Map;

import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.search.ui.ISearchResult;
import org.eclipse.search.ui.ISearchResultListener;
import org.eclipse.search.ui.ISearchResultPage;
import org.eclipse.search.ui.ISearchResultViewPart;
import org.eclipse.search.ui.SearchResultEvent;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.part.Page;
import org.hibernate.criterion.MatchMode;

import alma.acs.tmcdb.BACIProperty;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ComponentType;
import alma.acs.tmcdb.Computer;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.DefaultCanAddress;
import alma.acs.tmcdb.translator.TmcdbObject;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.handlers.AbstractEditObjectAction;
import alma.obops.tmcdbgui.handlers.EditAssemblyAction;
import alma.obops.tmcdbgui.handlers.EditBACIPropertyAction;
import alma.obops.tmcdbgui.handlers.EditComponentAction;
import alma.obops.tmcdbgui.handlers.EditComputerAction;
import alma.obops.tmcdbgui.handlers.EditConfigurationAction;
import alma.obops.tmcdbgui.handlers.EditContainerAction;
import alma.obops.tmcdbgui.handlers.EditDefaultCanAddressAction;
import alma.obops.tmcdbgui.utils.ImageHelper;
import alma.obops.tmcdbgui.utils.conversation.AssemblyConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.BACIPropertyConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.ComponentConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.ComputerConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.ContainerConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.obops.tmcdbgui.views.providers.SwDeploymentTreeLabelProvider;
import alma.tmcdb.domain.Assembly;
import alma.tmcdb.domain.AssemblyType;
import alma.tmcdb.domain.HwConfiguration;

public class SearchViewPage extends Page implements ISearchResultPage, ISearchResultListener {

	public static class TmcdbSearchLabelProvider extends SwDeploymentTreeLabelProvider
			implements ITableLabelProvider {

		public TmcdbSearchLabelProvider() {
			super();
		}

		@Override
		public Image getImage(Object element) {
			if( element instanceof TmcdbQueryTables )
				return RcpUtils.getImage(((TmcdbQueryTables)element).getIconName());
			if( element instanceof Configuration ||
			    element instanceof ComponentType )
				return ImageHelper.getImage((TmcdbObject)element);
			return super.getImage(element);
		}

		@Override
		public String getText(Object element) {
			if( element instanceof TmcdbQueryTables )
				return ((TmcdbQueryTables)element).getNiceName();
			if( element instanceof ComponentType )
				return ((ComponentType)element).getIDL();
			if( element instanceof Configuration )
				return ((Configuration)element).getConfigurationName();
			return super.getText(element);
		}

		@Override
		public Image getColumnImage(Object element, int columnIndex) {
			if( columnIndex == 0 )
				return getImage(element);
			else
				if( !(element instanceof TmcdbQueryTables) ) {
					Configuration conf = getConfiguration(element);
					if( conf != null )
						return getImage(conf);
				}
			return null;
		}

		@Override
		public String getColumnText(Object element, int columnIndex) {

			if( columnIndex == 0 )
				return getText(element);

			else
				if( !(element instanceof TmcdbQueryTables) ) {
					Configuration conf = getConfiguration(element);
					if( conf != null )
						return conf.getConfigurationName();
				}

			return null;
		}

		private Configuration getConfiguration(final Object element) 
		{
			try {
				if( element instanceof Component ) {
					ComponentConversationUtils.getInstance().runWithAttachedObject(element, new Runnable() {
						public void run() {
							((Component)element).getConfiguration().getConfigurationName();
						}
					});
					return ((Component) element).getConfiguration();
				}

				else if( element instanceof Container ) {
					ContainerConversationUtils.getInstance().runWithAttachedObject(element, new Runnable() {
						public void run() {
							((Container)element).getConfiguration().getConfigurationName();
						}
					});
					return ((Container) element).getConfiguration();
				}

				else if( element instanceof Computer ) {
					ComputerConversationUtils.getInstance().runWithAttachedObject(element, new Runnable() {
						public void run() {
							((Computer)element).getConfiguration().getConfigurationName();
						}
					});
					return ((Computer) element).getConfiguration();
				}

				else if( element instanceof BACIProperty ) {
					BACIPropertyConversationUtils.getInstance().runWithAttachedObject(element, new Runnable() {
						public void run() {
							((BACIProperty)element).getComponent().getConfiguration().getConfigurationName();
						}
					});
					return ((BACIProperty) element).getComponent().getConfiguration();
				}

				else if( element instanceof Assembly ) {
					AssemblyConversationUtils.getInstance().runWithAttachedObject(element, new Runnable() {
						public void run() {
							((Assembly)element).getConfiguration().getSwConfiguration().getConfigurationName();
						}
					});
					return ((Assembly)element).getConfiguration().getSwConfiguration();
				}
				
				// these are globals
				else if( element instanceof Configuration ||
				         element instanceof ComponentType ||
				         element instanceof AssemblyType ) {
					return null;
				}
			}
			catch(Exception ex) {
				ex.printStackTrace();
			}

			return null;
		}
	}

	public static class TmcdbSearchContentProvider implements ITreeContentProvider {

		private Map<TmcdbQueryTables, List<Object>> elements;

		@Override
		public void dispose() {
		}

		@SuppressWarnings("unchecked")
		@Override
		public void inputChanged(Viewer theViewer, Object oldInput, Object newInput) {

			if( newInput == null )
				return;

			elements = (Map<TmcdbQueryTables, List<Object>>)newInput;
		}

		@Override
		public Object[] getElements(Object inputElement) {
			return elements.keySet().toArray();
		}

		@Override
		public Object[] getChildren(Object parentElement) {
			return elements.get(parentElement).toArray();
		}

		@Override
		public Object getParent(Object element) {
			return null;
		}

		@Override
		public boolean hasChildren(Object element) {
			if( element instanceof TmcdbQueryTables )
				return true;
			return false;
		}

	}

    private AbstractEditObjectAction editContainer;
    private AbstractEditObjectAction editBACIProperty;
    private AbstractEditObjectAction editComponent;
    private AbstractEditObjectAction editComputer;
    private AbstractEditObjectAction editDCA;
    private AbstractEditObjectAction editConfigurationAction;
    private EditAssemblyAction       editAssembly;

	private TreeViewer viewer;
	private String _id;
	ISearchResult _search;

	@Override
	public void createControl(Composite parent) {

		Tree tree = new Tree(parent, SWT.BORDER | SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL);
		TreeColumn col1 = new TreeColumn(tree, SWT.LEFT);
		col1.setText("Object");
		col1.setWidth(500);
		TreeColumn col2 = new TreeColumn(tree, SWT.LEFT);
		col2.setText("Configuration");
		col2.setWidth(200);
		tree.setHeaderVisible(true);

		viewer = new TreeViewer(tree);
		viewer.setLabelProvider(new TmcdbSearchLabelProvider());
		viewer.setContentProvider(new TmcdbSearchContentProvider());
		viewer.setSorter(new ViewerSorter());

		makeActions();
	}

	private void makeActions() {

		IWorkbenchWindow win = getSite().getWorkbenchWindow();

		editContainer    = new EditContainerAction( win );
		editComponent    = new EditComponentAction( win );
		editComputer     = new EditComputerAction( win );
		editBACIProperty = new EditBACIPropertyAction(win);
		editDCA          = new EditDefaultCanAddressAction( win );
		editAssembly     = new EditAssemblyAction( win );
		editConfigurationAction = new EditConfigurationAction(win, null);

		viewer.addDoubleClickListener(new IDoubleClickListener() {
			public void doubleClick(DoubleClickEvent event) {
				if( event.getSelection() instanceof IStructuredSelection ) {
					IStructuredSelection selection = (IStructuredSelection)event.getSelection();

					if( selection.getFirstElement() instanceof BACIProperty )
						editBACIProperty.runWithDoubleClick(selection);
					if( selection.getFirstElement() instanceof Component )
						editComponent.runWithDoubleClick(selection);
					if( selection.getFirstElement() instanceof Computer )
						editComputer.runWithDoubleClick(selection);
					if( selection.getFirstElement() instanceof Container )
						editContainer.runWithDoubleClick(selection);
					if( selection.getFirstElement() instanceof DefaultCanAddress )
						editDCA.runWithDoubleClick(selection);
					if( selection.getFirstElement() instanceof Configuration ) {
						Configuration conf = (Configuration) selection.getFirstElement();
						List<HwConfiguration> l = null;
						try {
							l = HwConfigurationConversationUtils.getInstance().findConfigurationsByName(conf.getConfigurationName(), MatchMode.EXACT);
						} catch (Exception e) {
							// shouldn't happen, but who knows...
							e.printStackTrace();
							return;
						}
						editConfigurationAction.runWithDoubleClick(new StructuredSelection(l.get(0)));
					}
					if( selection.getFirstElement() instanceof DefaultCanAddress ) {
						editAssembly.selectionChanged(getSite().getPage().getActivePart(), selection);
						editAssembly.run();
					}
				}
			}
		});
	}

	@Override
	public Control getControl() {
		return viewer.getControl();
	}

	@Override
	public void setFocus() { }

	@Override
	public Object getUIState() {
		return null;
	}

	@Override
	public void setInput(ISearchResult search, Object uiState) {

		if( search == null )
			return;

		if( _search != null )
			_search.removeListener(this);

		_search = search;
		_search.addListener(this);
	}

	@Override
	public void setViewPart(ISearchResultViewPart part) {}

	@Override
	public void restoreState(IMemento memento) { }

	@Override
	public void saveState(IMemento memento) { }

	@Override
	public void setID(String id) {
		_id = id;
	}

	@Override
	public String getID() {
		return _id;
	}

	@Override
	public String getLabel() {
		return "TMCDB objects";
	}

	@Override
	public void searchResultChanged(SearchResultEvent e) {
		Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				viewer.setInput(((TmcdbSearchResult)_search).getElements());
			}
		});
	}

}
