/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * ResulSetUpdater.java
 *
 * Copyright European Southern Observatory 2008
 */

package alma.obops.tmcdbgui.rsviewer;

import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Types;
import java.util.ArrayList;
import java.util.List;

import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.utils.HibernateUtils;


/**
 * Handles change requests for a result set.
 *
 * @author amchavan, Sep 2, 2008
 * 
 */



public class ResultSetUpdater implements ResultSetCallback {

    protected ResultSet resultSet;
    protected HibernateUtils hibernateUtils;
    protected List<ChangeRequest> requests;
    protected ResultSetMetaData rsMeta;
    protected int numResSetCols;
    protected String tableName;
    protected List<String> keyIndexes;
    protected String whereClause;

    /**
     * Represents a change request for a result set: row, column and new value.
     * Minimal implementation with public member variables (like a C "struct").
     * Declared as 'static' so it can be used outside of this class (e.g.
     * for unit tests).
     */
    public static class ChangeRequest {

        public int row;
        public int column;
        public String newValue;

        public ChangeRequest( int row, int column, String newValue ) {
            this.row = row;
            this.column = column;
            this.newValue = newValue;
        }
    }

    /**
     * Constructor.
     * 
     * @param connection
     *            The JDBC database connection
     * 
     * @param resultSet
     *            The result set to update. It is assumed that all columns of
     *            the result set belong to the same database table, and that all
     *            columns needed to build a primary key for that table are
     *            included in the result set.<br/>
     * 
     *            If those conditions are not met, most likely some ugly errors
     *            will result from trying to update the result set. Ugly and
     *            potentially dangerous...
     *            
     * TODO Check that we are given a valid result set
     * 
     * @throws SQLException
     */
    public ResultSetUpdater( String tableName, HibernateUtils hibernateUtils, ResultSet resultSet ) 
        throws SQLException {
        
        this.resultSet = resultSet;
        this.hibernateUtils = hibernateUtils;
        this.requests = new ArrayList<ChangeRequest>();
        this.rsMeta = resultSet.getMetaData();
        this.numResSetCols = rsMeta.getColumnCount();
        this.tableName = tableName;
        this.keyIndexes = TmcdbContextFactory.INSTANCE.getHibernateUtils().getPrimaryKeyColumnNames(null, tableName);
    }

    /**
     * We implement the ResultSetCallback interface by simply storing a change
     * request for later attention.
     * 
     * @see alma.obops.rpcsandbox.spreadsheet.ResultSetCallback#updateResultSet(int,int,java.lang.String)
     */
    public void updateResultSet( int row, int column, String newValue ) {
        System.out.println( 
            ">>> updateRowColumn(" + row + "," + column + "," + newValue + ")");
        
        ChangeRequest request = new ChangeRequest( row, column, newValue );
        requests.add( request );
    }

    /**
     * Converts the input change request into an equivalent SQL statement
     * @throws SQLException 
     */
    public String processUpdateRequest( ChangeRequest request ) throws SQLException {

        System.out.print( ">>> processUpdateRequest(" + request.row + ","
                + request.column + "," + request.newValue + "): " );
        
        String colNname = rsMeta.getColumnName( request.column );

        StringBuilder sb = new StringBuilder();
        sb.append( "update " )
          .append( this.tableName )
          .append( " set " )
          .append( colNname )
          .append( "=" );

        if( request.newValue == ResultSetViewer.SQL_NULL ) {
            sb.append( "NULL" );
        }
        else if( isNumericColumn( request.column )) {
            sb.append( request.newValue );
        }
        else {
            sb.append( "'" ).append( request.newValue ).append( "'" );
        }
        
        String where = getWhereClause( request.row );
        sb.append( " where " ).append( where );
        
        String sql = sb.toString();
        System.out.println( sql );
        return sql;
    }

    /**
     * Process all outstanding change requests by attempting to modify the
     * underlying database (according to the request itself); each successfully
     * processed request is then eliminated from the pending queue.
     * 
     * @throws SQLException
     */
    public void update() throws SQLException {
        int count = getPendingRequestCount();
        for( int i = 0; i < count; i++ ) {
            updateOne();
        }
    }

    /**
     * Process the first outstanding change requests by attempting to modify the
     * underlying database (according to the request itself); the successfully
     * processed request is then eliminated from the pending queue.
     * @throws SQLException
     */
    public void updateOne() throws SQLException {
        ChangeRequest request = requests.get( 0 );
        //String sql = 
        	processUpdateRequest( request );
     // TODO uncomment this
//        this.hibernateUtils.runSql( sql );
        requests.remove( 0 );
    }
    
    /**
     * @return the number of pending change requests
     */
    public int getPendingRequestCount() {
        return requests.size();
    }
    
//    /**
//     * @return <code>true</code> if all columns in the res;
//     *         <code>false</code> otherwise.
//     * @throws SQLException
//     */
//    protected boolean onlyOneTable() throws SQLException {
//        String tableName = rsMeta.getTableName( 1 );
//        for( int i = 2; i < numCols + 1; i ++ ) {
//            if( ! rsMeta.getTableName( i ).equals( tableName )) {
//                return false;
//            }
//        }
//        return true;
//    }
    
    /**
     * Build a WHERE clause
     * @param row  The result set row to identify
     * @return The WHERE clause uniquely identifying a table row
     */
    public String getWhereClause( int row ) throws SQLException {
        resultSet.beforeFirst();
        for( int i = 0; i < row; i++ ) {
            resultSet.next();
        }
        StringBuilder sb = new StringBuilder();
        for( int i = 0; i < keyIndexes.size(); i++ ) {
            if( i > 0 ) {
                sb.append( " and " );
            }
            String colName = keyIndexes.get(i);
            sb.append( colName ).append( "=" );
            String colValue = resultSet.getString(colName);
            if( isNumericColumn( i )) {
                sb.append( colValue );
            }
            else {
                sb.append( "'" ).append( colValue ).append( "'" );
            }
        }
        return sb.toString();
    }

    /**
     * @return <code>true</code> if the input column is of some numeric type,
     *         <code>false</code> otherwise.
     * @throws SQLException
     */
    protected boolean isNumericColumn( int column ) throws SQLException {
      int columntype = rsMeta.getColumnType( column );
      boolean numeric = columntype == Types.BIGINT
              || columntype == Types.DECIMAL
              || columntype == Types.DOUBLE
              || columntype == Types.FLOAT
              || columntype == Types.INTEGER
              || columntype == Types.NUMERIC
              || columntype == Types.REAL
              || columntype == Types.SMALLINT
              || columntype == Types.TINYINT;
      return numeric;
    }
}
