/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.rcp;

import org.eclipse.jface.action.GroupMarker;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.action.ICoolBarManager;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.action.ToolBarManager;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.actions.ActionFactory.IWorkbenchAction;
import org.eclipse.ui.actions.ContributionItemFactory;
import org.eclipse.ui.application.ActionBarAdvisor;
import org.eclipse.ui.application.IActionBarConfigurer;
import org.hibernate.criterion.MatchMode;

import alma.acs.tmcdb.Configuration;
import alma.obops.tmcdb.alarms.ui.actions.add.AddAlarmCategoryAction;
import alma.obops.tmcdb.alarms.ui.actions.add.AddDefaultMemberAction;
import alma.obops.tmcdb.alarms.ui.actions.add.AddFaultCodeAction;
import alma.obops.tmcdb.alarms.ui.actions.add.AddFaultFamilyAction;
import alma.obops.tmcdb.alarms.ui.actions.add.AddFaultMemberAction;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.handlers.AddAntennaAction;
import alma.obops.tmcdbgui.handlers.AddAssemblyAction;
import alma.obops.tmcdbgui.handlers.AddFrontendAction;
import alma.obops.tmcdbgui.handlers.AddHolographyTowerAction;
import alma.obops.tmcdbgui.handlers.AddPadAction;
import alma.obops.tmcdbgui.handlers.AddWeatherStationAction;
import alma.obops.tmcdbgui.handlers.CloneAntennaAction;
import alma.obops.tmcdbgui.handlers.CloneConfigurationAction;
import alma.obops.tmcdbgui.handlers.ExportConfigurationAction;
import alma.obops.tmcdbgui.handlers.ImportConfigurationAction;
import alma.obops.tmcdbgui.handlers.NewBACIPropertyAction;
import alma.obops.tmcdbgui.handlers.NewComponentAction;
import alma.obops.tmcdbgui.handlers.NewComputerAction;
import alma.obops.tmcdbgui.handlers.NewContainerAction;
import alma.obops.tmcdbgui.handlers.QueryConfigurationsAction;
import alma.obops.tmcdbgui.handlers.ShowDefaultCanAddressesAction;
import alma.obops.tmcdbgui.handlers.ShowDeviceLibrariesAction;
import alma.obops.tmcdbgui.handlers.ShowNotificationChannelsAction;
import alma.obops.tmcdbgui.handlers.ShowSQLConsoleAction;
import alma.obops.tmcdbgui.handlers.ShowSwDeploymentAction;
import alma.obops.tmcdbgui.views.ConfigurationsView;
import alma.obops.tmcdbgui.views.IConfigurationSearcher;

/**
 * An action bar advisor is responsible for creating, adding, and disposing of
 * the actions added to a workbench window. Each window will be populated with
 * new actions.
 * 
 * @author amchavan, Sep 11, 2008 -- from the wizard
 * 
 */


public class ApplicationActionBarAdvisor extends ActionBarAdvisor implements IModelChangeListener, IConfigurationSearcher
{
    private static final String MB_ADDITIONS = "additions";

    // Actions - important to allocate these only in makeActions, and then use
    // them in the fill methods. This ensures that the actions aren't recreated
    // when fillActionBars is called with FILL_PROXY.
    private IAction exitAction;
    private IAction rpAction;
    private IAction exportConfigAction;
    private IAction showSwDeployAction;
    private IAction showNCsAction;
	private IAction showDCAAction;
    private IAction showDeviceLibrariesAction;
    private IAction showSQLConsoleAction;
    private IAction addAntennaAction;
    private IAction addPadAction;
    private IAction addFrontendAction;
    private IAction addAssemblyAction;
    private IAction addWeatherStationAction;
    private IAction addHolographyTowerAction;
    private IAction importConfigAction;
    private IAction cloneConfigAction;
    private IAction cloneAntennaAction;
    private IAction saveEditorAction;
    private IAction deleteRetargetableAction;
    private IAction aboutAction;
    private IAction queryConfigurationsAction;
    
    private AddAlarmCategoryAction newAlarmCategoryAction;
    private AddFaultCodeAction newFaultCodeAction;
    private AddFaultFamilyAction newFaultFamilyAction;
    private AddFaultMemberAction newFaultMemberAction;
    private AddDefaultMemberAction newDefaultMemberAction;
    
    private NewBACIPropertyAction newBaciPropertyAction;
    private NewComponentAction newComponentAction;
    private NewContainerAction newContainerAction;
    private NewComputerAction newComputerAction;
    
    //private IContributionItem addAntennaCommandContributionItem; 
    
    private IContributionItem perspectivesMenu;
    private IContributionItem viewsMenu;

	private IWorkbenchAction preferncesAction;
    
    public ApplicationActionBarAdvisor( IActionBarConfigurer configurer ) {
        super( configurer );
    }

    protected void makeActions( final IWorkbenchWindow window ) {
        // Creates the actions and registers them.
        // Registering is needed to ensure that key bindings work.
        // The corresponding commands key bindings are defined in the plugin.xml
        // file.
        // Registering also provides automatic disposal of the actions when
        // the window is closed.

    	/* Actions taken from other plug-ins */
        exitAction = ActionFactory.QUIT.create( window );
        register( exitAction );
        
        rpAction = ActionFactory.RESET_PERSPECTIVE.create( window );
        register( rpAction );

        saveEditorAction = ActionFactory.SAVE.create(window);
        register(saveEditorAction);

        deleteRetargetableAction = ActionFactory.DELETE.create(window);
        register(deleteRetargetableAction);

        aboutAction = ActionFactory.ABOUT.create(window);
        register(aboutAction);

        /* Our own actions */
        CloneConfigurationAction hwaction1 = new CloneConfigurationAction(window, this);
        cloneConfigAction = hwaction1;
        register(cloneConfigAction);

        CloneAntennaAction hwaction2 = new CloneAntennaAction(window, this);
        cloneAntennaAction = hwaction2;
        register(cloneAntennaAction);

        AddAntennaAction hwaction3 = new AddAntennaAction(window, this);
        addAntennaAction = hwaction3;
        register(addAntennaAction);
        
        AddAssemblyAction hwaction4 = new AddAssemblyAction(window, this);
        addAssemblyAction = hwaction4;
        register(addAssemblyAction);
        
        AddFrontendAction hwaction5 = new AddFrontendAction(window, this);
        addFrontendAction = hwaction5;
        register(addFrontendAction);
        
        AddPadAction hwaction6 = new AddPadAction(window, this); 
        addPadAction = hwaction6;       	
        register(addPadAction);
        
        AddHolographyTowerAction hwaction7 = new AddHolographyTowerAction(window); 
        addHolographyTowerAction = hwaction7;       	
        register(addHolographyTowerAction);
        
        AddWeatherStationAction hwaction8 = new AddWeatherStationAction(window); 
        addWeatherStationAction = hwaction8;       	
        register(addWeatherStationAction);
        
        exportConfigAction = new ExportConfigurationAction(window);
        register(exportConfigAction);
        
        queryConfigurationsAction = new QueryConfigurationsAction(window);
        register(queryConfigurationsAction);

        ImportConfigurationAction impConfAction = new ImportConfigurationAction(window, this);
        impConfAction.addModelChangeListener(this);
        importConfigAction = impConfAction;
        register(importConfigAction);

        showSwDeployAction = new ShowSwDeploymentAction(window);
        register(showSwDeployAction);

        showNCsAction = new ShowNotificationChannelsAction(window);
        register(showNCsAction);

        showDCAAction = new ShowDefaultCanAddressesAction( window );
        register(showDCAAction);

        showDeviceLibrariesAction = new ShowDeviceLibrariesAction();
        register(showDeviceLibrariesAction);

        showSQLConsoleAction = new ShowSQLConsoleAction();
        register(showSQLConsoleAction);
        
        newBaciPropertyAction = new NewBACIPropertyAction(window, null);
        register(newBaciPropertyAction);
        
        newComponentAction = new NewComponentAction(window, null);
        register(newComponentAction);
        
        newContainerAction = new NewContainerAction(window, null);
        register(newContainerAction);
        
        newComputerAction = new NewComputerAction(window, null);
        register(newComputerAction);
        
        newAlarmCategoryAction = new AddAlarmCategoryAction(window, null);
        register(newAlarmCategoryAction);
        
        newFaultFamilyAction = new AddFaultFamilyAction(window, null, null);
        register(newFaultFamilyAction);
        
        newFaultCodeAction = new AddFaultCodeAction(window, null, null);
        register(newFaultCodeAction);
        
        newFaultMemberAction = new AddFaultMemberAction(window, null, null);
        register(newFaultMemberAction);
        
        newDefaultMemberAction = new AddDefaultMemberAction(window);
        register(newDefaultMemberAction);

        // Perspective and View switch menus as the ones in Eclipse under "Window"
        perspectivesMenu =  ContributionItemFactory.PERSPECTIVES_SHORTLIST.create( window );
        viewsMenu =  ContributionItemFactory.VIEWS_SHORTLIST.create(window);     
        preferncesAction = ActionFactory.PREFERENCES.create(window);
    }

    protected void fillMenuBar( IMenuManager menuBar ) {

    	/* Explorer */
        MenuManager fileMenu = new MenuManager( "&Explorer", "explorer" );        
        fileMenu.add( saveEditorAction );
        fileMenu.add( new Separator() );
        fileMenu.add( showSQLConsoleAction );
        fileMenu.add( showDeviceLibrariesAction );
        fileMenu.add( exitAction );

        /* Configuration */
        MenuManager configurationNewMenu = new MenuManager("&New", "configuration-new");
        configurationNewMenu.add( addAntennaAction );
        configurationNewMenu.add( addAssemblyAction );
        configurationNewMenu.add( addFrontendAction );
        configurationNewMenu.add( addHolographyTowerAction );
        configurationNewMenu.add( addPadAction );
        configurationNewMenu.add( addWeatherStationAction );
        configurationNewMenu.add( newBaciPropertyAction );
        configurationNewMenu.add( newComponentAction );
        configurationNewMenu.add( newContainerAction );
        configurationNewMenu.add( newComputerAction );
        configurationNewMenu.add( new Separator(MB_ADDITIONS) );
        
        MenuManager alarmsMenu = new MenuManager("&Alarms", "configuration-alarms");
        alarmsMenu.add( newAlarmCategoryAction );
        alarmsMenu.add( newFaultFamilyAction );
        alarmsMenu.add( newFaultMemberAction );
        alarmsMenu.add( newDefaultMemberAction );
        alarmsMenu.add( newFaultCodeAction );
        configurationNewMenu.add(alarmsMenu);

        MenuManager configurationShowMenu = new MenuManager("&Show", "configuration-show");
        configurationShowMenu.add( showSwDeployAction );
        configurationShowMenu.add( showNCsAction );
        configurationShowMenu.add( showDCAAction );

        MenuManager configurationMenu = new MenuManager("&Configuration","configuration");
        configurationMenu.add( configurationNewMenu );
        configurationMenu.add( configurationShowMenu );
        configurationMenu.add( new Separator() );
        configurationMenu.add( queryConfigurationsAction );
        configurationMenu.add( cloneConfigAction );
        configurationMenu.add( cloneAntennaAction );
        configurationMenu.add( new Separator() );
        configurationMenu.add( importConfigAction );
        configurationMenu.add( exportConfigAction );

        /* Window */
        MenuManager perspectivesMM = new MenuManager( "Show &Perspectives", "perspectives" );
        perspectivesMM.add( perspectivesMenu );
        perspectivesMM.add( rpAction );

        MenuManager viewsMM = new MenuManager( "Show &Views", "views" );
        viewsMM.add(viewsMenu);

        MenuManager windowMM = new MenuManager("&Window", "window");
        windowMM.add( perspectivesMM );
        windowMM.add( viewsMM );
        windowMM.add( preferncesAction );

        MenuManager helpMenu = new MenuManager("&Help", "help");
        helpMenu.add( aboutAction );

        /* Finally, the top-level menu */
        menuBar.add( fileMenu );
        menuBar.add( configurationMenu );
        menuBar.add( windowMM );
        menuBar.add( helpMenu );
    }

	@Override
	protected void fillCoolBar(ICoolBarManager coolBar) {

		IToolBarManager newObjectsToolbar = new ToolBarManager(coolBar.getStyle());
		newObjectsToolbar.add( addAntennaAction );
		newObjectsToolbar.add( addAssemblyAction );
		newObjectsToolbar.add( addFrontendAction );
		newObjectsToolbar.add( addHolographyTowerAction );
		newObjectsToolbar.add( addPadAction );
		newObjectsToolbar.add( addWeatherStationAction );
		newObjectsToolbar.add( newBaciPropertyAction );
		newObjectsToolbar.add( newComponentAction );
		newObjectsToolbar.add( newContainerAction );
		newObjectsToolbar.add( newComputerAction );
		newObjectsToolbar.add( newAlarmCategoryAction );
		newObjectsToolbar.add( newFaultFamilyAction );
		newObjectsToolbar.add( newFaultMemberAction );
		newObjectsToolbar.add( newDefaultMemberAction );
		newObjectsToolbar.add( newFaultCodeAction );

		IToolBarManager saveDeleteObjectsToolbar = new ToolBarManager(coolBar.getStyle());
		saveDeleteObjectsToolbar.add( saveEditorAction );
		saveDeleteObjectsToolbar.add( deleteRetargetableAction );

		IToolBarManager configObjectsToolbar = new ToolBarManager(coolBar.getStyle());
		configObjectsToolbar.add( queryConfigurationsAction );
		
		coolBar.add( configObjectsToolbar );
		coolBar.add( saveDeleteObjectsToolbar );
		coolBar.add( newObjectsToolbar );
		coolBar.add( new GroupMarker(MB_ADDITIONS) );
	}

	public AddAntennaAction getAddAntennaAction() {
		return (AddAntennaAction)this.addAntennaAction;
	}
	
	public AddPadAction getAddPadAction() {
		return (AddPadAction)this.addPadAction;
	}
	
	public AddWeatherStationAction getAddWeatherStationAction() {
		return (AddWeatherStationAction)this.addWeatherStationAction;
	}
	
	public AddHolographyTowerAction getAddHolographyTowerAction() {
		return (AddHolographyTowerAction)this.addHolographyTowerAction;
	}
	
	public AddFrontendAction getAddFrontendAction() {
		return (AddFrontendAction)this.addFrontendAction;
	}
	
	public AddAssemblyAction getAddAssemblyAction() {
		return (AddAssemblyAction)this.addAssemblyAction;
	}
	
	public NewComputerAction getNewComputerAction() {
		return this.newComputerAction;
	}
	
	public NewContainerAction getNewContainerAction() {
		return this.newContainerAction;
	}
	
	public NewComponentAction getNewComponentAction() {
		return this.newComponentAction;
	}
	
	public NewBACIPropertyAction getNewBACIPropertyAction() {
		return this.newBaciPropertyAction;
	}

	public QueryConfigurationsAction getQueryConfigurationsAction() {
		return (QueryConfigurationsAction)this.queryConfigurationsAction;
	}
	
	public AddAlarmCategoryAction getAddAlarmCategoryAction() {
		return this.newAlarmCategoryAction;
	}
	
	public AddFaultFamilyAction getFaultFamilyAction() {
		return this.newFaultFamilyAction;
	}
	
	public AddFaultMemberAction getFaultMemberAction() {
		return this.newFaultMemberAction;
	}
	
	public AddDefaultMemberAction getDefaultMemberAction() {
		return this.newDefaultMemberAction;
	}
	
	public AddFaultCodeAction getFaultCodeAction() {
		return this.newFaultCodeAction;
	}
	
	@Override
	public void externalModelChange() {
		// this is a bit of a hack; we can't register the listeners directly on
		// the configurations view at the time this action bar adviser is created, 
		// because the views don't yet exist (haven't been instantiated)! 
		// NOTE: if we add actions on the toolbar / menus that affect the startup 
		// scenario view, we'll need to do something similar for the startupscenario view.
		ConfigurationsView configsView = (ConfigurationsView)RcpUtils.findView(ConfigurationsView.ID);
		configsView.externalModelChange();
	}

	@Override
	public void internalModelChange() {
		// this is a bit of a hack; we can't register the listeners directly on
		// the configurations view at the time this action bar adviser is created, 
		// because the views don't yet exist (haven't been instantiated)! 
		// NOTE: if we add actions on the toolbar / menus that affect the startup 
		// scenario view, we'll need to do something similar for the startupscenario view.
		ConfigurationsView configsView = (ConfigurationsView)RcpUtils.findView(ConfigurationsView.ID);
		configsView.internalModelChange();
	}

	@Override
	public void setSearchCriteria(String configurationName, boolean queryAllActiveStates, boolean active, MatchMode matchMode) {
		// this is a bit of a hack; we can't register the listeners directly on
		// the configurations view at the time this action bar adviser is created, 
		// because the views don't yet exist (haven't been instantiated)! 
		// NOTE: if we add actions on the toolbar / menus that affect the startup 
		// scenario view, we'll need to do something similar for the startupscenario view.
		ConfigurationsView configsView = (ConfigurationsView)RcpUtils.findView(ConfigurationsView.ID);
		if(null != configsView) {
			configsView.setSearchCriteria(configurationName, queryAllActiveStates, active, matchMode);
		}
	}

	/**
	 * Used to update some of the actions to denote that 
	 * a new configuration is being manipulated.
	 * 
	 * @param config the new configuration being manipulated.
	 */
	public void configurationChanged(Configuration config) {
		this.newBaciPropertyAction.setConfiguration(config);
		this.newComponentAction.setConfiguration(config);
		this.newComputerAction.setConfiguration(config);
		this.newContainerAction.setConfiguration(config);
		this.newAlarmCategoryAction.setConfiguration(config);
		this.newFaultFamilyAction.setConfiguration(config);
	}
}