/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * 
 */
package alma.obops.tmcdbgui.observablevalues;

import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.databinding.observable.Diffs;
import org.eclipse.core.databinding.observable.value.AbstractObservableValue;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;

import alma.acs.tmcdb.ContainerRealTimeType;

/**
 * Observable value used to observe a group of radio buttons. The values observed are the strings that are set
 * through the {@link Button#setText(String)} method, unless they are explicitly passed through the constructor.
 * This is used across several editors of the TMCDB Explorer where a radio button group is used to choose among
 * a little amount of options.
 *  
 * @author rtobar, Mar 22, 2010
 *
 */
public class RadioButtonsToContainerRealTimeTypeObservableValue extends AbstractObservableValue {

	private ContainerRealTimeType oldValue = null;
	Map<Button, ContainerRealTimeType> radios;
	
	/**
	 * Creates an observable value for a set of radio buttons. If given, the values
	 * and buttons must be provided in the same order as they correlate.
	 * 
	 * @param values The values that will be observed. If null, the string returned
	 *   by the {@link Button#getText()} method of the corresponding button is used
	 *   as the value being observed.
	 * @param buttons The radio buttons that represent these values
	 */
	public RadioButtonsToContainerRealTimeTypeObservableValue(ContainerRealTimeType [] values, Button ...buttons) {

		if( values != null && buttons.length != values.length )
			throw new RuntimeException("Number of values and buttons to be observed should be the same");

		radios = new HashMap<Button, ContainerRealTimeType>();

		Listener listener = new Listener() {
			public void handleEvent(Event event) {
				if( ((Button)event.widget).getSelection() ) {
					ContainerRealTimeType value = (ContainerRealTimeType)doGetValue();
					if( oldValue == null || !oldValue.equals(value) ) {
						fireValueChange(Diffs.createValueDiff(oldValue, value));
						oldValue = value;
					}
				}
			}
		};

		for(int i=0; i!= buttons.length; i++) {
			buttons[i].addListener(SWT.Selection, listener);
			if( values != null )
				radios.put(buttons[i], values[i]);
			else
				radios.put(buttons[i], ContainerRealTimeType.valueOfForEnum(buttons[i].getText()));
		}

	}

	/* (non-Javadoc)
	 * @see org.eclipse.core.databinding.observable.value.AbstractObservableValue#doGetValue()
	 */
	@Override
	protected Object doGetValue() {
		for (Button button : radios.keySet()) {
			if( button.getSelection() )
				return radios.get(button);
		}
		return null;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.core.databinding.observable.value.IObservableValue#getValueType()
	 */
	@Override
	public Object getValueType() {
		return ContainerRealTimeType.class;
	}

	public void doSetValue(Object o) {
		ContainerRealTimeType s = (ContainerRealTimeType)o;
		for(Button radio: radios.keySet()) {
			ContainerRealTimeType name = radios.get(radio);
			if( name.equals(s) ) {
				radio.setSelection(true);
				return;
			}
		}
	}
}
