/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.observablevalues;

import java.util.Calendar;
import java.util.Date;

import org.eclipse.core.databinding.observable.Diffs;
import org.eclipse.core.databinding.observable.value.AbstractObservableValue;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.DateTime;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;

/**
 * A wrapper allowing DateTime widget to be observed.
 * @author Moritz Post
 * @see http://eclipsesource.com/blogs/2009/02/03/databinding-a-custom-observable-for-your-widget/
 */
public class DateTimeObservableValue extends AbstractObservableValue 
{

	private final DateTime dateTime;

	protected Date oldValue;

	Listener listener = new Listener() {
		@Override
		public void handleEvent(final Event event) {
			Date newValue = dateTimeToDate(dateTime);

			if (!newValue.equals(DateTimeObservableValue.this.oldValue)) {
				fireValueChange(Diffs.createValueDiff(DateTimeObservableValue.this.oldValue,
						newValue));
				DateTimeObservableValue.this.oldValue = newValue;

			}
		}

	};

	public DateTimeObservableValue(final DateTime dateTime) {
		this.dateTime = dateTime;
		this.dateTime.addListener(SWT.Selection, this.listener);
	}

	@Override
	protected Object doGetValue() {
		return dateTimeToDate(dateTime);
	}

	@Override
	protected void doSetValue(final Object value) {
		if (value instanceof Date) {
			Date date = (Date) value;
			dateToDateTime(date, dateTime);
		}
	}

	@Override
	public Object getValueType() {
		return Date.class;
	}
	
	public static void dateToDateTime(final Date date, DateTime dateTimeToSet) {
		if (!dateTimeToSet.isDisposed()) {
			Calendar cal = Calendar.getInstance();
			cal.setTime(date);
			dateTimeToSet.setYear(cal.get(Calendar.YEAR));
			dateTimeToSet.setMonth(cal.get(Calendar.MONTH));
			dateTimeToSet.setDay(cal.get(Calendar.DAY_OF_MONTH));
			dateTimeToSet.setHours(cal.get(Calendar.HOUR_OF_DAY));
			dateTimeToSet.setMinutes(cal.get(Calendar.MINUTE));
			dateTimeToSet.setSeconds(cal.get(Calendar.SECOND));
		}
	}

	public static Date dateTimeToDate(DateTime dt) {
		Date result = null;
		if (!dt.isDisposed()) {
			Calendar cal = Calendar.getInstance();
			cal.set(Calendar.YEAR, dt.getYear());
			cal.set(Calendar.MONTH, dt.getMonth());
			cal.set(Calendar.DAY_OF_MONTH, dt.getDay());
			cal.set(Calendar.HOUR_OF_DAY, dt.getHours());
			cal.set(Calendar.MINUTE, dt.getMinutes());
			cal.set(Calendar.SECOND, dt.getSeconds());
			result = cal.getTime();
		}
		return result;
	}

	@Override
	public synchronized void dispose() {
		this.dateTime.removeListener(SWT.Selection, this.listener);
		super.dispose();
	}

}
