/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.ActionFactory.IWorkbenchAction;

import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.handlers.conversation.ModelPublisherAction;
import alma.obops.tmcdbgui.utils.conversation.BackendConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.BaseElementConversationUtils;
import alma.obops.tmcdbgui.views.ConfigurationsView;
import alma.tmcdb.domain.HolographyTower;
import alma.tmcdb.domain.HolographyTowerToPad;

/**
 * Removes a HolographyTowerToPad assignment from a holography tower.
 * @author sharring
 */
public class UnassignPadFromHolographyTowerAction extends
		ModelPublisherAction implements ISelectionListener, IWorkbenchAction 
{
	private IStructuredSelection selection;
	private IWorkbenchWindow window;
	private String ID = "alma.obops.tmcdbgui.handlers.UnassignPadFromHolographyTowerAction";
	
	public UnassignPadFromHolographyTowerAction(IWorkbenchWindow window, IModelChangeListener listener) 
    {
    	this.window = window;
    	setId(ID);
        setText("Unassign");
		setToolTipText("Removes a pad to holography tower assignment");
    	window.getSelectionService().addSelectionListener(this);
    	this.addModelChangeListener(listener);
    }
	
	@Override
	public void selectionChanged(IWorkbenchPart win, ISelection newSelection) 
	{
		if( newSelection instanceof IStructuredSelection) 
		{
			selection = (IStructuredSelection)newSelection;
			setEnabled( selection.size() == 1 &&
			            selection.getFirstElement() instanceof HolographyTowerToPad);	
		}
	}

	@Override
	public void dispose() {
		window.getSelectionService().removeSelectionListener(this);
	}
	
	public void run() 
	{
		HolographyTowerToPad h2p = (HolographyTowerToPad)selection.getFirstElement();
		HolographyTower tower = h2p.getHolographyTower();
		boolean removed = tower.getAssociatedPads().remove(h2p);
		
		// BEGIN HACK
		if(!removed) 
		{
			// in case the remove operation failed inappropriately (hashCode / equals problems?) 
			// we will iterate to remove the item by brute force, which should not be necessary
			HolographyTowerToPad h2pToRemove = null;
			for(HolographyTowerToPad h2pIterated : tower.getAssociatedPads())
			{
				if(h2pIterated.getHolographyTowerToPadId().equals(h2p.getHolographyTowerToPadId()))
				{
					h2pToRemove = h2pIterated;
					break;
				}
			}
			if(null != h2pToRemove)
			{
				boolean removedAgain = tower.getAssociatedPads().remove(h2pToRemove);
				if(!removedAgain) 
				{
					throw new IllegalStateException("Could not remove pad to holography tower assignment!");
				}
			}
		}
		// END HACK
		try {
			BackendConversationUtils.getInstance().delete(h2p, ConversationToken.CONVERSATION_COMPLETED);
			// TODO: make this call use ConversationToken.CONVERSATION_PENDING && refactor other methods so that
			// only the *final* call uses CONVERSATION_COMPLETED (so that this action is atomic and either
			// succeeds completely or fails completely)!
			BaseElementConversationUtils.getInstance().saveOrUpdateHolographyTower(tower);
			this.modelShouldBeReloaded();
			ConfigurationsView configView = (ConfigurationsView)RcpUtils.findView(ConfigurationsView.ID);
			configView.getConfigurationsTreeViewer().refresh();
		} catch (Exception e) {
			throw new RuntimeException("Could not update pad to holography tower assignment", e);
		}
	}
}
