/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import java.lang.reflect.InvocationTargetException;
import java.sql.SQLException;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IViewActionDelegate;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.PlatformUI;

import alma.obops.tmcdbgui.TmcdbGui;
import alma.obops.tmcdbgui.rsviewer.ResultSetUpdater;
import alma.obops.tmcdbgui.views.RawDataView;

/**
 * Loops over the current change list, executing an <code>UPDATE</code> SQL
 * statements for each change.
 * 
 * @author amchavan, Sep 5, 2008
 * 
 */



public class SaveChangesAction implements IViewActionDelegate {
    
    public static final String ID = "saveChanges.action";
    
    /** Sleep for some number of milliseconds */
    protected static void sleep( long msec ) {
        try {
            Thread.sleep( msec );
        }
        catch( Exception e ) {
            e.printStackTrace();        // what else?
        }
    }
    private SQLException lastError;

    protected RawDataView view;
    
    /**
     * Save the view for future reference
     *
     * @see org.eclipse.ui.IViewActionDelegate#init(org.eclipse.ui.IViewPart)
     */
    public void init( IViewPart theView) {
        this.view = (RawDataView) theView;
    }


    /**
     * Loops over the current change list, executing an <code>UPDATE</code> SQL
     * statements for each change.
     * 
     * @see org.eclipse.ui.IActionDelegate#run(org.eclipse.jface.action.IAction)
     */
    public void run( IAction action ) {        
        final ResultSetUpdater updater = TmcdbGui.getResultSetUpdater();
        
        // do we have anything to do?
        if( updater == null || updater.getPendingRequestCount() == 0 ) {
            return;
        }
        
        this.lastError = null;
        final int count = updater.getPendingRequestCount();
        final ProgressMonitorDialog pd = new ProgressMonitorDialog( null );
        final Display display = PlatformUI.getWorkbench().getDisplay();
        
        try {
            pd.run(  true, true, new IRunnableWithProgress() {

                public void run( IProgressMonitor monitor )
                    throws InvocationTargetException, InterruptedException {
                    
                    // What follows is boilerplate code adapted from 
                    // the RCP book, managing the progress bar
                    //-----------------------------------------------
                    String msg = "Saving " + count + " changes to the TMCDB";
                    monitor.beginTask( msg, count + 1 );
                    for( int i = 0; i < count; i++ ) {
                        sleep( 500 );   // so that user can see some progress
                        if( monitor.isCanceled() ) {
                            return;
                        }
                        final int current = i+1;
                        monitor.subTask( "Saving change: " + current );
                        display.asyncExec( new Runnable() {

                            // This is where we do a real "work item":
                            // save the next change request and save any
                            // errors for later display
                            //------------------------------------------
                            public void run() {
                                try {
                                    updater.updateOne();
                                }
                                catch( SQLException e ) {
                                    e.printStackTrace();
                                    saveError( e );
                                }
                            }
                        });
                        monitor.worked( 1 );

                        try {
                        	view.reload();
                        } catch(Exception e) {
                        	e.printStackTrace();
                        }
                        monitor.worked(1);
                        monitor.done();
                    }
                }
            });
        }
        catch( Exception e ) {
            e.printStackTrace();
        }
        if( lastError != null ) {
            IStatus status = new Status( IStatus.ERROR, 
                                         "tmcdbguircp", 
                                         0, 
                                         lastError.getCause().getMessage(), 
                                         lastError.getCause() );
            // IStatus status = new Status( IStatus.ERROR, "tmcdbguircp",
            // e.getMessage() );
            Shell shell = view.getSite().getShell();
            ErrorDialog.openError( shell, 
                                   "SQL Error", 
                                   "Failed SQL: " + lastError.getMessage(),
                                   status );
        }
        return;
    }


    protected void saveError( SQLException ex ) {
        this.lastError = ex;
    }

    /**
     * No-op
     * @see org.eclipse.ui.IActionDelegate#selectionChanged(org.eclipse.jface.action.IAction, org.eclipse.jface.viewers.ISelection)
     */
    public void selectionChanged( IAction action, ISelection selection ) {
        // no-op
    }
}
