/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.hibernate.exception.ConstraintViolationException;

import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AssemblyConversationUtils;
import alma.tmcdb.domain.Assembly;

/**
 * Action to rename (i.e. change the serial number for) an assembly.
 * NOTE: the motivation for this action emanated from COMP-4945.
 * 
 * @author sharring
 */
public class RenameAssemblyAction extends ConversationalAction 
{
	private static final String RENAME_FAILED = "Rename failed";
	private static final String UNEXPECTED_EXCEPTION = "Unexpected exception";
	private static final String S_N_ALREADY_IN_USE = "S/N already in use; must be unique (within configuration)";
	private static final String ENTER_NEW_SERIAL_NUMBER_FOR_ASSEMBLY = "Enter new serial number for assembly";
	private static final String RENAME_ASSEMBLY = "Rename Assembly";
	
	private String ID = "rename_assembly.action";
	private IWorkbenchWindow _window;
	private String newSerialNumber = null;
	private Assembly assembly;
	
	public RenameAssemblyAction(IWorkbenchWindow win,
			IModelChangeListener listener) 
	{
		_window = win;
		setId(ID);
		setText("&" + RENAME_ASSEMBLY);
		setToolTipText(RENAME_ASSEMBLY);
		_window.getSelectionService().addSelectionListener(this);
		this.addModelChangeListener(listener);
	}

	@Override
	public void doPreConversational() 
	{
		InputDialog dlg = new InputDialog(Display.getCurrent().getActiveShell(),
				RENAME_ASSEMBLY, ENTER_NEW_SERIAL_NUMBER_FOR_ASSEMBLY, null, null);

		if (dlg.open() != Window.CANCEL) {
			newSerialNumber = dlg.getValue();
		}
	}
	
	@Override
	public void doConversational() 
	{
		if(null != newSerialNumber )
		{
			String oldAssemblySerialNumber = assembly.getSerialNumber();
			try {
				// set a wait cursor
				_window.getShell().setCursor(_window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
				assembly.setSerialNumber(newSerialNumber);
				AssemblyConversationUtils.getInstance().saveOrUpdateAssembly(assembly);
			} 
			catch(ConstraintViolationException e) {
				String message = S_N_ALREADY_IN_USE;
				rollbackChange(oldAssemblySerialNumber, message);
			}
			catch (Exception e) {
				String message = UNEXPECTED_EXCEPTION;
				rollbackChange(oldAssemblySerialNumber, message);
				e.printStackTrace();
			}
			finally {
				// set a wait cursor
				_window.getShell().setCursor(null);
			}
		}
	}

	private void rollbackChange(String oldSsn, String message) 
	{
		assembly.setSerialNumber(oldSsn);
		MessageDialog.openWarning(_window.getShell(), RENAME_FAILED, message);
	}

	@Override
	public void doPostConversational() 
	{
		this.modelChanged();
	}

	@Override
	public void selectionChanged(IWorkbenchPart part, ISelection newSelection) 
	{
		setEnabled(false);
		if( newSelection instanceof IStructuredSelection && GuiUtils.isGodUser()) 
		{
			selection = (IStructuredSelection)newSelection;
			if(selection.size() == 1 && (selection.getFirstElement() instanceof Assembly))	
			{
				assembly = (Assembly) (selection.getFirstElement());
				setEnabled(true);
			}
		}
	}
}
