/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.DomainObjectUtils;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.StartupScenarioConversationUtils;
import alma.obops.tmcdbgui.views.StartupScenariosView;
import alma.tmcdb.domain.BaseElementStartup;

/**
 * Action to remove a base element startup from a startup scenario.
 * @author sharring
 */
public class RemoveBaseElementStartupAction extends ConversationalAction 
{
	private static final String REMOVE_BASE_ELEMENT_STARTUPS = "Remove Base Element Startup(s)";
	private static final String REMOVE_BASE_ELEMENT_STARTUP = "Remove Base Element Startup";
	private static final String ID = "remove_baseelementstartup.action";
	private BaseElementStartup[] beStartups;
	

	/**
	 * Constructor.
	 * @param window the window associated with this action.
	 */
	public RemoveBaseElementStartupAction(IWorkbenchWindow window, IModelChangeListener listener) 
	{
		this.window = window;
		this.listeners.add(listener);

		setId(ID);
		setText( REMOVE_BASE_ELEMENT_STARTUPS );
	}

	@Override
	public void doConversational() {
		Shell shell = window.getShell();

		try
		{
			for(BaseElementStartup bes: beStartups) 
			{
				BaseElementStartup rootOfTree = DomainObjectUtils.determineRootOfBaseElementTree(bes);
				StartupScenarioConversationUtils.getInstance().removeBaseElementFromStartupScenario(bes, 
						rootOfTree.getStartup());
			}
		}
		catch( Exception e ) {
			e.printStackTrace();
			MessageDialog.openError( shell, 
					"Removal of Base Element Startup",
					e.getClass().getSimpleName() + 
					": " + e.getMessage() );
		} finally {
			shell.setCursor(null);
		}

	}
	
	@Override
	public void doPostConversational() 
	{
		Shell shell = window.getShell();
		// set a wait cursor
		shell.setCursor(shell.getDisplay().getSystemCursor(SWT.CURSOR_WAIT));

		try {
			for(BaseElementStartup bes: beStartups) {
				// Notify listeners (e.g. the view) so they can 
				// re-display the configuration with the new scenario
				StartupScenariosView view = (StartupScenariosView)RcpUtils.findView(StartupScenariosView.ID);
				view.removeObjectFromTree(bes);
			}
		}
		finally {
			// unset wait cursor
			shell.setCursor(null);
		}
	}

	@Override
	public void selectionChanged(IWorkbenchPart part, ISelection newSelection) 
	{
		if( newSelection instanceof IStructuredSelection && GuiUtils.isGodUser()) 
		{
			selection = (IStructuredSelection)newSelection;
			if( selection.size() >= 1 &&
					 GuiUtils.onlyItemsOfClassSelected(selection, BaseElementStartup.class) )	
			{
				setEnabled(true);
				this.beStartups = (BaseElementStartup[]) selection.toArray();
			}
			else {
				setEnabled(false);
			}
		}
		else {
			setEnabled(false);
		}

	}

	public void setBaseElementStartups(BaseElementStartup[] baseElementStartupsSelected) 
	{
		this.beStartups = baseElementStartupsSelected;
		if(beStartups.length > 1) {
			setText(REMOVE_BASE_ELEMENT_STARTUPS);
		} else {
			setText(REMOVE_BASE_ELEMENT_STARTUP);
		}
	}
}
