/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import java.util.List;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.ActionFactory.IWorkbenchAction;
import org.hibernate.criterion.MatchMode;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.obops.tmcdbgui.views.ConfigurationsView;
import alma.obops.tmcdbgui.wizards.QueryConfigurationsWizard;
import alma.tmcdb.domain.HwConfiguration;

/**
 * Reload the list of configurations
 *
 * @author amchavan, Sep 5, 2008
 * 
 */



public class QueryConfigurationsAction extends Action implements ISelectionListener, IWorkbenchAction
{
    public static final String ID = "query-configurations.action";
	private String configurationName;
	private boolean activeOnly;
	private boolean queryAllActiveStates;
	private MatchMode matchMode = MatchMode.ANYWHERE;
	private List<HwConfiguration> configurations;
	private IWorkbenchWindow _window;

	public QueryConfigurationsAction(IWorkbenchWindow window) {
		_window = window;
		setId(ID);
		setText("&Query configurations...");
		setToolTipText("Queries configurations from the database");
		setImageDescriptor(RcpUtils.getImageDescriptor("icons/find.png"));
		window.getSelectionService().addSelectionListener(this);
		setEnabled(true);
	}

	@Override
	public void run() {
		doConversational();
		doPostConversational();
	}

	@Override
	public void selectionChanged(IWorkbenchPart part, ISelection selection) {
	}
	
	@Override
	public void dispose() {
		_window.getSelectionService().removeSelectionListener(this);
	}
	
	public static List<HwConfiguration> queryConfigsByName(Shell shell, String configurationName, boolean allActiveStates, boolean active, MatchMode matchMode) 
	{
		List<HwConfiguration> configs = null;
		try {
			if(allActiveStates) {
				configs = HwConfigurationConversationUtils.getInstance().findConfigurationsByName(configurationName, matchMode);
			}
			else {
				configs = HwConfigurationConversationUtils.getInstance().findConfigurationsByName( configurationName, active, matchMode );
			}
			if( configs.size() == 0 ) {
				MessageDialog.openInformation( shell, "Query information", 
				"No configurations found" );
			} 
		}
		catch( Exception e ) {
			e.printStackTrace();
			MessageDialog.openError( shell, "Query failed", e.getMessage() );
		} 
		return configs;
	}

	/** Called by our wizard */
	public void setConfigurationName( String configurationName ) {
		this.configurationName = configurationName;
	}
	
	public void setActiveOnly( boolean activeOnly ) {
		this.activeOnly = activeOnly;
	}
	
	public void setMatchMode( MatchMode matchMode ) {
		this.matchMode = matchMode;
	}
	
	public void setQueryAllActiveStates( boolean queryAllActiveStates ) {
		this.queryAllActiveStates = queryAllActiveStates;
	}
	
	public void doPostConversational() {
		ConfigurationsView view = (ConfigurationsView) RcpUtils.findView(ConfigurationsView.ID);
		view.setSearchCriteria(configurationName, queryAllActiveStates, activeOnly, matchMode);
		view.setInput( configurations.toArray(new HwConfiguration[0]) );
	}

	public void doConversational()
	{
		Shell shell = _window.getShell();

		// Collect user input -- the wizard will set the configurationName
		// field if user clicks on the "Finish" button
		QueryConfigurationsWizard wizard = new QueryConfigurationsWizard( this );
		WizardDialog dialog = new WizardDialog( shell, wizard );
		int ret = dialog.open();
		if( ret == WizardDialog.CANCEL ) {
			return;
		}

		if(null != configurationName && configurationName.trim().length() > 0) {
			RcpUtils.statusMessage( "Configuration name: " + configurationName );
		} else {
			RcpUtils.statusMessage( "Configuration name: *");
		}
		
		try {
			shell.setCursor(shell.getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			configurations = queryConfigsByName(shell, configurationName, queryAllActiveStates, activeOnly, matchMode);
		} finally {
			shell.setCursor(null);
		}
	}
}
