/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * NewAntennaAction.java
 *
 * Copyright European Southern Observatory 2008
 */

package alma.obops.tmcdbgui.handlers;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.StartupScenarioConversationUtils;
import alma.obops.tmcdbgui.wizards.NewStartupScenarioWizard;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.StartupScenario;

/**
 * Creates a new StartupScenario, attached to some configuration
 *
 * @author amchavan, Sep 12, 2008
 * 
 */



public class NewStartupScenarioAction extends ConversationalAction 
{
	private static final String ID = "add_startup.action";
	protected HwConfiguration configuration;
    private String name;

    /**
     * Constructor.
     * @param window the window associated with this action.
     */
    public NewStartupScenarioAction(IWorkbenchWindow window) 
    {
    	this.window = window;
		setId(ID);
    	setText( "New Startup Scenario" );
        setImageDescriptor( RcpUtils.getImageDescriptor( "icons/startup.png" ));
    }
    
    /**
     * Setter for the configuration in which the new startup scenario
     * will exist / be added.
     * 
     * @param configuration the configuration to which the new startup
     * scenario will be added.
     */
    public void setConfiguration(HwConfiguration configuration) 
    {
    	this.configuration = configuration;
    }

    /**
     * Getter for the name of the new startup scenario.
     * @return name the name of the new startup scenario.
     */
    public String getName() {
        return name;
    }
    
    /**
     * Setter for the name of the startup scenario that will be added.
     * @param name the name of the new startup scenario.
     */
    public void setName( String name ) {
        this.name = name;
    }

    @Override
    public void doConversational() 
    {	
    	Shell shell = window.getShell();
        
        try {
            // Collect user input
            NewStartupScenarioWizard wizard = new NewStartupScenarioWizard( this );
            WizardDialog dialog = new WizardDialog( shell, wizard );
            int ret = dialog.open();
            if( ret == WizardDialog.CANCEL ) {
                return;
            }
            
            // Create and save a startup with the given info
            StartupScenario newStartup = 
                new StartupScenario(  name );
            newStartup.setConfiguration(configuration);

            configuration.addStartupScenario( newStartup );
            shell.setCursor(shell.getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
            StartupScenarioConversationUtils.getInstance().saveOrUpdateStartupScenario( newStartup );
        }
        catch( Exception e ) {
            e.printStackTrace();
            MessageDialog.openError( shell, 
                                     "Creation of Startup Scenario",
                                     e.getClass().getSimpleName() + 
                                     ": " + e.getMessage() );
        } finally {
        	shell.setCursor(null);
        }
    }

	@Override
	public void doPostConversational() {
		Shell shell = window.getShell();
		try {
			shell.setCursor(shell.getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			
			// Notify listeners (e.g. the view) so they can 
			// re-display the configuration with the new scenario
			modelChanged();
		}
		finally {
			shell.setCursor(null);
		}
	}

	@Override
	public void selectionChanged(IWorkbenchPart part, ISelection newSelection) 
	{
		if( newSelection instanceof IStructuredSelection ) 
		{
			selection = (IStructuredSelection)newSelection;
			if( selection.size() == 1 && GuiUtils.isGodUser() &&
					   (selection.getFirstElement() instanceof HwConfiguration) )	
			{
				setEnabled(true);
				this.configuration = (HwConfiguration) selection.getFirstElement();
			}
			else {
				setEnabled(false);
			}
		}
		else {
			setEnabled(false);
		}
	}
}
