/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) NRAO - National Radio Astronomy Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/

package alma.obops.tmcdbgui.handlers;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.ActionFactory.IWorkbenchAction;

import alma.acs.tmcdb.AcsService;
import alma.acs.tmcdb.AcsServiceServiceType;
import alma.acs.tmcdb.ChannelMapping;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.NotificationServiceMapping;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AcsServiceConversationUtils;
import alma.obops.tmcdbgui.wizards.NewChannelMappingWizard;

public class NewChannelMappingAction extends Action implements
		ISelectionListener, IWorkbenchAction 
{
	private IStructuredSelection _selection;
    private IWorkbenchWindow _window;
    private final static String ID = "new_channelmapping.action";
    private NotificationServiceMapping nsMapping;

    public NewChannelMappingAction() {}

    public NewChannelMappingAction(IWorkbenchWindow window) 
    {
    	_window = window;
    	_window.getSelectionService().addSelectionListener(this);
    	setEnabled(GuiUtils.isGodUser());
    	setId(ID);
    	this.setEnabled(GuiUtils.isGodUser());
    	this.setToolTipText("Creates a new Notification Service / Channel Mapping definition");
    	this.setText("New Channel Mapping...");
    	this.setImageDescriptor(RcpUtils.getImageDescriptor("icons/new-channel-mapping.gif"));
    }

	@Override
	public void run() 
	{
		String[] nsStrings = getNotificationServiceStrings(nsMapping.getConfiguration());
		if(null == nsStrings || nsStrings.length == 0) {
			informUserOfMissingNotificationServices(_window.getShell());
		}
		else {
			NewChannelMappingWizard wizard = new NewChannelMappingWizard(nsMapping, nsStrings);
	    	wizard.init(_window.getWorkbench(), _selection);
	    	WizardDialog dialog = new WizardDialog(_window.getShell(), wizard);
	    	dialog.create();
	    	dialog.open();
		}
	}

	public static void informUserOfMissingNotificationServices(Shell shell) {
		MessageDialog.openInformation(shell, 
				"No notification services exist", 
				"No custom notification services exist in this configuration. \n\nTo create a notification service, browse to the computer which will host the service, right click, choose 'New ACS service', and create a service of type 'notification'");
	}

	public static String[] getNotificationServiceStrings(Configuration config) 
	{
		String [] retVal = null;
		try 
		{
			AcsServiceConversationUtils.getInstance().hydrateAcsServices(config);
			List<AcsService> acsServices = new ArrayList<AcsService>();
			for(AcsService service : config.getAcsServices())
			{
				if(service.getServiceType().equals(AcsServiceServiceType.NOTIFICATION))
				{
					acsServices.add(service);
				}
			}
			
			retVal = new String[acsServices.size()];
			int count = 0;
			for(AcsService service: acsServices)
			{
				retVal[count++] = service.getServiceInstanceName();
			}
		} 
		catch (Exception e) 
		{
			throw new RuntimeException("Could not get names of notification services");
		}
		return retVal;
	}
	
	@Override
	public void dispose() 
	{
		_window.getSelectionService().removeSelectionListener(this);
	}

	@Override
	public void selectionChanged(IWorkbenchPart part, ISelection selection) 
	{
		if( selection instanceof IStructuredSelection && GuiUtils.isGodUser()) 
		{
			_selection = (IStructuredSelection)selection;
			if( _selection.getFirstElement() instanceof ChannelMapping[]) 
			{
				setEnabled(true);
				nsMapping = (((ChannelMapping[])_selection.getFirstElement())[0]).getNotificationServiceMapping();
			}
			else if(_selection.getFirstElement() instanceof NotificationServiceMapping) {
				setEnabled(true);
				nsMapping = (NotificationServiceMapping)_selection.getFirstElement();
			}
			else {
				setEnabled(false);
			}
		}
		else {
			setEnabled(false);
		}
	}
}
