/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.ActionFactory.IWorkbenchAction;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;
import org.eclipse.ui.dialogs.SelectionDialog;

import alma.acs.tmcdb.Computer;
import alma.acs.tmcdb.Configuration;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.dialogs.ComputerSelectionDialog;
import alma.obops.tmcdbgui.dialogs.ComputerSelectionDialogLabelProvider;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.views.support.AcsServiceWrapper;
import alma.obops.tmcdbgui.wizards.NewAcsServiceWizard;

/**
 * Used to create a new acs service when/if the user has selected the "Services" node in the main tree.
 *
 * @author sharring
 *
 * @see alma.obops.tmcdbgui.views.support.AcsServiceWrapper
 */
public class NewAcsServiceWhenWrapperSelectedAction extends Action implements ISelectionListener, IWorkbenchAction 
{
    private IStructuredSelection _selection;
    private IWorkbenchWindow _window;
    private final static String ID = "new_service_wrapper.action";
    private Computer computer;
    private Configuration configuration;

    public NewAcsServiceWhenWrapperSelectedAction() {}

    public NewAcsServiceWhenWrapperSelectedAction(IWorkbenchWindow window, Configuration config) 
    {
    	_window = window;
    	_window.getSelectionService().addSelectionListener(this);
    	configuration = config; 
    	setEnabled(GuiUtils.isGodUser());
    	setId(ID);
    	this.setEnabled(GuiUtils.isGodUser());
    	this.setToolTipText("Creates a new ACS Service definition");
    	this.setText("New ACS Service...");
    	this.setImageDescriptor(RcpUtils.getImageDescriptor("icons/new_service.png"));
    }

	@Override
	public void run() 
	{
		computer = getTargetComputer();
		if(null == computer) {
			return;
		}
		NewAcsServiceWizard wizard = new NewAcsServiceWizard(computer, configuration);
    	wizard.init(_window.getWorkbench(), _selection);
    	WizardDialog dialog = new WizardDialog(_window.getShell(), wizard);
    	dialog.create();
    	dialog.open();
	}

	@Override
	public void dispose() 
	{
		_window.getSelectionService().removeSelectionListener(this);
	}

	@Override
	public void selectionChanged(IWorkbenchPart part, ISelection selection) 
	{
		if( selection instanceof IStructuredSelection && GuiUtils.isGodUser()) 
		{
			_selection = (IStructuredSelection)selection;
			if( _selection.getFirstElement() instanceof AcsServiceWrapper[]) 
			{
				setEnabled(true);
			}
			else {
				setEnabled(false);
			}
		} else {
			setEnabled(false);
		}
	}

	public void setConfiguration(Configuration swConfiguration) {
		this.configuration = swConfiguration;
	}
	
	private Computer getTargetComputer() 
	{
		ElementListSelectionDialog d = new ComputerSelectionDialog(_window.getShell(), new ComputerSelectionDialogLabelProvider(), configuration);
		if( d.open() == SelectionDialog.CANCEL )
			return null;

		Object computers[] = d.getResult();
		if( computers != null && computers.length == 1 ) {
			return (Computer)computers[0];
		}

		return null;
	}
}

