/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import java.io.FileInputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.Date;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.ui.IWorkbenchWindow;
import org.hibernate.criterion.MatchMode;

import alma.obops.dam.tmcdb.domain.TMCDBExport;
import alma.obops.dam.utils.xstream.GlobalConfigurationUnmarshallingException;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.handlers.conversation.ModelPublisherAction;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.obops.tmcdbgui.views.IConfigurationSearcher;
import alma.obops.utils.DatetimeUtils;
import alma.obops.utils.DatetimeUtils.Style;

public class ImportConfigurationAction extends ModelPublisherAction {

	private IWorkbenchWindow _window;
	private String ID = "import_configuration.action";
	private IConfigurationSearcher configSearcher;
	private String newName;

	public ImportConfigurationAction(IWorkbenchWindow window, IConfigurationSearcher searcher) 
	{
		_window = window;
		setId(ID);
		setText("&Import configuration...");
		setToolTipText("Imports a configuration from a XML file");
		setImageDescriptor(RcpUtils.getImageDescriptor("icons/import.gif"));
		this.configSearcher = searcher;
	}

	@Override
	public void run() 
	{
		String fileNameTmp = null;
		FileDialog fileDialog = new FileDialog(_window.getShell(), SWT.OPEN);
		fileDialog.setFilterExtensions(new String[] { "*.*", "*.xml" });
		fileDialog.setFileName("configuration.xml");
		fileDialog.setOverwrite(true);
		fileNameTmp = fileDialog.open();

		if( fileNameTmp == null )
			return;

		final String fileName = fileNameTmp;
		ProgressMonitorDialog pd = new ProgressMonitorDialog(_window.getShell());
		try {
			pd.run(true, true, new ImportRunnable(fileName));
				
		} catch (InvocationTargetException e1) {
			e1.printStackTrace();
		} catch (InterruptedException e1) {
			e1.printStackTrace();
		}
		
		configSearcher.setSearchCriteria(newName, true, false, MatchMode.EXACT);
		modelShouldBeReloaded();	
	}
	
	
	private class ImportRunnable implements IRunnableWithProgress
    {
		private String fileName;
		
		public ImportRunnable(String fileName)
		{
			this.fileName = fileName;
		}
		
		public void run(IProgressMonitor monitor) throws InvocationTargetException,
				InterruptedException 
		{
			monitor.beginTask("Configuration XML import", 100);
			monitor.worked(1);

			// Step 1: Open and read the file
			if( monitor.isCanceled() )
				return;
			String xml = null;
			monitor.subTask("Reading " + fileName);
			try {
				FileInputStream fis = new FileInputStream(fileName);
				byte[] b = new byte[fis.available()];
				fis.read(b);
				fis.close();
				xml = new String(b);
			} catch(Exception e) {
				RcpUtils.errorMessage(e, _window.getShell(), "Import failed",
						"Unexpected exception while reading " + fileName);
				return;
			}
			monitor.worked(15);

			// Step 2: Import configuration from XML
			if( monitor.isCanceled() )
				return;
			TMCDBExport conf = null;
			monitor.subTask("Importing configuration from XML");
			try {
				conf = HwConfigurationConversationUtils.getInstance().importConfigurationFromXml(xml);
			} 
			catch(Exception ex) 
			{
				
				Throwable ultimateCause = ex;
				while(ultimateCause.getCause() != null) {
					ultimateCause = ultimateCause.getCause();
				}
				if(ultimateCause instanceof GlobalConfigurationUnmarshallingException)
				{
					RcpUtils.errorMessage(_window.getShell(), "Problem with global configuration reference(s).",
							ultimateCause.getMessage());
				}
				else 
				{
					RcpUtils.errorMessage(ex, _window.getShell(), "Import failed",
					"Unexpected exception while importing the configuration");
				}
				return;
			}
			monitor.worked(30);

			// Step 3: clone imported configuration
			if( monitor.isCanceled() )
				return;
			monitor.subTask("Saving imported Configuration");
			try {
				newName = conf.get_hwconfig().getName()+"-imported-"+DatetimeUtils.getIsoDateFormat(Style.SHORTER).format(new Date());
				HwConfigurationConversationUtils.getInstance().cloneImportedConfiguration(conf, newName);
			} catch (Exception e) {
				RcpUtils.errorMessage(e, _window.getShell(), "Import failed",
						"Unexpected exception while cloning imported configuration");
			}
			monitor.worked(53);
		}
	}
}
