/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import java.io.FileWriter;
import java.lang.reflect.InvocationTargetException;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.ActionFactory.IWorkbenchAction;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.tmcdb.domain.HwConfiguration;

public class ExportConfigurationAction extends Action implements ISelectionListener, IWorkbenchAction {

	private IWorkbenchWindow _window;
	private IStructuredSelection _selection;
	private String ID = "export_configuration.action";

	public ExportConfigurationAction(IWorkbenchWindow window) {
		_window = window;
		setId(ID);
		setText("&Export configuration...");
		setToolTipText("Exports a configuration to a XML file");
		setImageDescriptor(RcpUtils.getImageDescriptor("icons/export.gif"));
		window.getSelectionService().addSelectionListener(this);
	}

	@Override
	public void run() {

		String fileNameTmp = null;
		FileDialog fileDialog = new FileDialog(_window.getShell(), SWT.SAVE);
		fileDialog.setFilterExtensions(new String[] { "*.*", "*.xml" });
		fileDialog.setFileName("configuration.xml");
		fileDialog.setOverwrite(true);
		fileNameTmp = fileDialog.open();

		if( fileNameTmp == null )
			return;

		final String fileName = fileNameTmp;
		ProgressMonitorDialog pd = new ProgressMonitorDialog(_window.getShell());
		try {
			pd.run(true, true,
				new IRunnableWithProgress() {
					public void run(IProgressMonitor monitor) throws InvocationTargetException,
							InterruptedException {

						monitor.beginTask("Configuration XML export", 100);
						monitor.worked(1);

						// Step 1: Hydrate the configuration
						if( monitor.isCanceled() )
							return;
						monitor.subTask("Hydrating configuration");
						HwConfiguration conf = (HwConfiguration)_selection.getFirstElement();
						try {
							conf = HwConfigurationConversationUtils.getInstance().hydrateConfigurationForExport(conf);
						} catch (Exception e) {
							final Exception fE = e;
							_window.getShell().getDisplay().asyncExec( new Runnable() {
								public void run() {
									MessageDialog.openError(_window.getShell(),
											"Export failed",
									"Unexpected exception while hydrating the configuration");
									fE.printStackTrace();
								}
							});
							return;
						}
						monitor.worked(79);

						// Step 2: Export configuration as XML
						if( monitor.isCanceled() )
							return;
						monitor.subTask("Exporting configuration as XML");
						String xml = null;
						try {

							// As we need to export the configuration using one of the proxies
							// instances of the HwConfiguration object, and not the object itself,
							// we need to check through the child elements to get a proxy instance
							if(conf.getAssemblies().size() > 0)
								xml = HwConfigurationConversationUtils.getInstance().exportConfigurationAsXml(
										conf.getAssemblies().iterator().next().getConfiguration());
							else if(conf.getBaseElements().size() > 0)
								xml = HwConfigurationConversationUtils.getInstance().exportConfigurationAsXml(
										conf.getBaseElements().iterator().next().getConfiguration());
							else if(conf.getStartupScenarios().size() > 0)
								xml = HwConfigurationConversationUtils.getInstance().exportConfigurationAsXml(
										conf.getStartupScenarios().iterator().next().getConfiguration());
							else if(conf.getHwSchemas().size() > 0)
								xml = HwConfigurationConversationUtils.getInstance().exportConfigurationAsXml(
										conf.getHwSchemas().iterator().next().getConfiguration());

							// If there are no child elements, then we can safely export using the real object
							else
								xml = HwConfigurationConversationUtils.getInstance().exportConfigurationAsXml(conf);

						} catch (Exception e) {
							final Exception fE = e;
							e.printStackTrace();
							_window.getShell().getDisplay().asyncExec( new Runnable() {
								public void run() {
									MessageDialog.openError(_window.getShell(),
											"Export failed",
									"Unexpected exception while exporting the configuration");
									fE.printStackTrace();
									return;
								}
							});
						}
						monitor.worked(15);

						// Save XML as a file
						if( monitor.isCanceled() )
							return;
						monitor.subTask("Saving file");
						try {
							FileWriter f = new FileWriter(fileName);
							f.write(xml);
							f.close();
						} catch (Exception e) {
							MessageDialog.openError(_window.getShell(),
									"Export failed",
									"Unexpected exception during file saving");
							e.printStackTrace();
						}
						monitor.worked(5);

					}
				}
			);
		} catch (InvocationTargetException e1) {
			e1.printStackTrace();
		} catch (InterruptedException e1) {
			e1.printStackTrace();
		}

	}

	@Override
	public void selectionChanged(IWorkbenchPart part, ISelection selection) {
		if( selection instanceof IStructuredSelection && GuiUtils.isGodUser()) {
			_selection = (IStructuredSelection)selection;
			setEnabled( ((IStructuredSelection) selection).size() == 1 &&
					_selection.getFirstElement() instanceof HwConfiguration);
		}
		else
			setEnabled(false);
	}

	@Override
	public void dispose() {
		_window.getSelectionService().removeSelectionListener(this);
	}

}
