/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.ActionFactory.IWorkbenchAction;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;
import org.eclipse.ui.dialogs.SelectionDialog;

import alma.acs.tmcdb.AcsService;
import alma.acs.tmcdb.Computer;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.dialogs.ComputerSelectionDialog;
import alma.obops.tmcdbgui.dialogs.ComputerSelectionDialogLabelProvider;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.LabelHelper;
import alma.obops.tmcdbgui.utils.conversation.AcsServiceConversationUtils;
import alma.obops.tmcdbgui.views.SoftwareDeploymentView;
import alma.obops.tmcdbgui.views.support.AcsServiceWrapper;

/**
 * This class is used to deploy a "wrapped" tmcdb object; currently the only object which 
 * this will apply to is the AcsService, which might be wrapped as an AcsServiceWrapper. The
 * purpose of the wrapper is explained in the AcsServiceWrapper class.
 *
 * @author sharring
 *
 * @see alma.obops.tmcdb.views.support.AcsServiceWrapper
 */
public class DeployWrappedAcsServiceAction  extends Action implements ISelectionListener, IWorkbenchAction
{
	private IStructuredSelection _selection;

	protected List<AcsServiceWrapper> _selectedServices = new ArrayList<AcsServiceWrapper>();
	protected IWorkbenchWindow _window;
	private static final ImageDescriptor IMAGE_DESCRIPTOR = RcpUtils.getImageDescriptor("icons/deploy.gif");
	private static final String TEXT_DESCRIPTOR = "Re-deploy ACS Service...";
	private static final String TOOLTIPTEXT = "Deploys the ACS Service to a different computer";
	private static final String ID = "deploy_wrapped_service.action";

    public DeployWrappedAcsServiceAction(IWorkbenchWindow window) {
    	this._window = window;
    	this._window.getSelectionService().addSelectionListener(this);
    	this.setEnabled(GuiUtils.isGodUser());
    	this.setId(ID);
    	this.setToolTipText(TOOLTIPTEXT);
    	this.setText(TEXT_DESCRIPTOR);
    	this.setImageDescriptor(IMAGE_DESCRIPTOR);
    }

	protected Computer getTargetComputer() throws NoComputerSelectedException 
	{
		ElementListSelectionDialog d = new ComputerSelectionDialog(_window.getShell(), new ComputerSelectionDialogLabelProvider(), _selectedServices.get(0).getAcsService().getConfiguration());
		if( d.open() == SelectionDialog.CANCEL )
			throw new NoComputerSelectedException();

		Object computers[] = d.getResult();
		if( computers != null && computers.length == 1 ) {
			return (Computer)computers[0];
		}

		throw new NoComputerSelectedException();
	}

	protected boolean validateSelectionItem(@SuppressWarnings("unused") Object item) {
		return true;
	}

	@Override
	public void run() 
	{
		AcsService service = null;
		Computer targetComputer = null;
		try {
			targetComputer = getTargetComputer();
		} catch (NoComputerSelectedException e1) { 
			return;
		}

		try {
			for(int i = 0; i != _selectedServices.size(); i++) {
				service = _selectedServices.get(i).getAcsService();
				String oldComputerNetworkName = service.getComputer().getNetworkName();
				service.setComputer(targetComputer);
				AcsServiceConversationUtils.getInstance().moveAcsService(service, oldComputerNetworkName);
			}
		} catch(Exception e) {
			e.printStackTrace();
			MessageDialog.openError(_window.getShell(),
			    "Error while re-assigning ACS Service",
			    "There was an unexpected error while re-assigning ACS Service '" + LabelHelper.getAcsServiceLabel(service) + "'");
		}

		SoftwareDeploymentView view = (SoftwareDeploymentView) RcpUtils.findView(SoftwareDeploymentView.ID);
		view.internalModelChange();
	}
	
	@Override
	public void dispose() {
		_window.getSelectionService().removeSelectionListener(this);
	}

	@Override
	public void selectionChanged(IWorkbenchPart part, ISelection selection) {
		if( selection instanceof IStructuredSelection && GuiUtils.isGodUser()) 
		{
			_selectedServices.clear();
			_selection = (IStructuredSelection)selection;
			if( selectionOnlyOfClass(_selection, AcsServiceWrapper.class, _selectedServices) )
				setEnabled(true);
			else
				setEnabled(false);
		}
		else {
			setEnabled(false);
		}
	}

	@SuppressWarnings("unchecked")
	private <T> boolean selectionOnlyOfClass(IStructuredSelection selection, Class<T> clazz, List<T> objects) {

		if( selection.size() == 0 )
			return false;

		for (Iterator<?> it = selection.iterator(); it.hasNext();) {
			Object item = it.next();
			if( !item.getClass().equals(clazz) )
				return false;
			if( !validateSelectionItem(item) )
				return false;
			objects.add((T)item);
		}
		return true;
	}

	protected static class NoComputerSelectedException extends Exception {
		private static final long serialVersionUID = 6047075548594687255L;
	}
	
}
