/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.ActionFactory.IWorkbenchAction;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;

import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.Container;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.dialogs.ContainerSelectionDialog;
import alma.obops.tmcdbgui.dialogs.ContainerSelectionDialogLabelProvider;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.LabelHelper;
import alma.obops.tmcdbgui.utils.conversation.ComponentConversationUtils;
import alma.obops.tmcdbgui.views.SoftwareDeploymentView;

/**
 * Deploys a component to a container (used e.g. when drag-n-drop isn't convenient, via a context menu).
 * @author sharring
 */
public class DeployComponentAction extends Action implements
		ISelectionListener, IWorkbenchAction 
{
	private static final ImageDescriptor IMAGE_DESCRIPTOR = RcpUtils.getImageDescriptor("icons/deploy.gif");
	private static final String TEXT_DESCRIPTOR = "Deploy component...";
	private static final String TOOLTIPTEXT = "Deploys a component to a container";
	private IStructuredSelection _selection;
	private Component component;
	private IWorkbenchWindow _window;
	private static final String ID = "deploy_component.action";

	public DeployComponentAction() {}

    public DeployComponentAction(IWorkbenchWindow window) {
    	_window = window;
    	_window.getSelectionService().addSelectionListener(this);
    	 this.setEnabled(GuiUtils.isGodUser());
    	 this.setId(ID);
    	 this.setToolTipText(TOOLTIPTEXT);
    	 this.setText(TEXT_DESCRIPTOR);
    	 this.setImageDescriptor(IMAGE_DESCRIPTOR);
    }
    

	@Override
	public void run() 
	{
		ElementListSelectionDialog d = new ContainerSelectionDialog(_window.getShell(), new ContainerSelectionDialogLabelProvider(), component.getConfiguration());
		d.open();
		Object containers[] = d.getResult();
		if( containers != null && containers.length == 1 ) {
			component.setContainer( (Container)containers[0] );
		}

		try {
			ComponentConversationUtils.getInstance().saveOrUpdateComponent(component);
		} catch(Exception e) {
			e.printStackTrace();
			MessageDialog.openError(_window.getShell(),
			    "Error while assigning component",
			    "There was an unexpected error while assigning component '" + LabelHelper.getFullPath(component,false) + "'");
		}
		SoftwareDeploymentView view = (SoftwareDeploymentView) RcpUtils.findView(SoftwareDeploymentView.ID);
		view.internalModelChange();
	}
	
	@Override
	public void dispose() {
		_window.getSelectionService().removeSelectionListener(this);
	}

	@Override
	public void selectionChanged(IWorkbenchPart part, ISelection selection) {
		if( selection instanceof IStructuredSelection && GuiUtils.isGodUser()) 
		{
			this.component = null;
			_selection = (IStructuredSelection)selection;
			if(_selection.getFirstElement() instanceof Component) 
			{
				this.component = (Component) _selection.getFirstElement();
				if(component.getContainer() == null) {
					setEnabled(true);
				} else {
					setEnabled(false);
				}
			}
			else {
				setEnabled(false);
			}
		}
		else {
			setEnabled(false);
		}
	}
}
