/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.ActionFactory.IWorkbenchAction;

import alma.acs.tmcdb.AcsService;
import alma.acs.tmcdb.BACIProperty;
import alma.acs.tmcdb.ChannelMapping;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.Computer;
import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.ContainerStartupOption;
import alma.acs.tmcdb.DomainsMapping;
import alma.acs.tmcdb.translator.TmcdbObject;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.obops.tmcdbgui.handlers.conversation.ModelPublisherAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.LabelHelper;
import alma.obops.tmcdbgui.utils.conversation.AcsServiceConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.BackendConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.ComponentConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.ComputerConversationUtils;
import alma.obops.tmcdbgui.widgets.DeleteMessageDialog;

public class DeleteSwDeploymentObjectAction extends ModelPublisherAction implements ISelectionListener, IWorkbenchAction {

	private IWorkbenchWindow _window;
	private IStructuredSelection _selection;
	private String ID = "alma.obops.tmcdbgui.handlers.DeleteSwDeploymentObjectAction";
	
	public DeleteSwDeploymentObjectAction(IWorkbenchWindow window) {
		_window = window;
		setId(ID);
		setText("&Delete");
		setToolTipText("Deletes the object");
		setImageDescriptor( PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(ISharedImages.IMG_ETOOL_DELETE) );
		window.getSelectionService().addSelectionListener(this);
	}

	@Override
	public void run() {

		List<TmcdbObject> objectsToDelete;

		objectsToDelete = getEligibleElementsForDeletion(_selection);
		if( objectsToDelete.size() == 0 ) {
			MessageDialog.openInformation(_window.getShell(), "No candidates for deletion",
			    "There are no eligible elements to be deleted, nothing will be done");
			return;
		}

		MessageDialog dialog = new DeleteMessageDialog(_window, objectsToDelete.toArray());
		if( dialog.open() != 0 )
			return;

		Iterator<?> it = objectsToDelete.iterator();

		while(it.hasNext()) {
			Object o = it.next();
			try {
				BackendConversationUtils.getInstance().delete(o, ConversationToken.CONVERSATION_COMPLETED);
				if(o instanceof AcsService) {
					AcsServiceConversationUtils.getInstance().updateManagerRecordsAfterAcsServiceDeletion((AcsService)o);
				}
				modelChanged();
			} catch (Exception e) {
				e.printStackTrace();
				MessageDialog.openError(_window.getShell(), "Error while deleting",
						"Error while deleting " + getMessageForObject(new StructuredSelection(o)) + ": " + e);
			}
		}

	}

	private String getMessageForObject(IStructuredSelection selection2) {

		Iterator<?> it = selection2.iterator();
		StringBuilder sb = new StringBuilder();

		while( it.hasNext() ) {

			Object selection = it.next();

			if( selection instanceof AcsService ) {
				AcsService serv = (AcsService)selection;
				sb.append("\nACS Service '" + LabelHelper.getAcsServiceLabel(serv) + "'");
			}
			else if( selection instanceof BACIProperty ) {
				BACIProperty bp = (BACIProperty)selection;
				sb.append("\nBACI Property '" + LabelHelper.getFullPath(bp, false) + "'");
			}
			else if( selection instanceof Component ) {
				Component c = (Component)selection;
				sb.append("\nComponent '" + LabelHelper.getFullPath(c, false) + "'");
			}
			else if( selection instanceof Container ) {
				Container c = (Container)selection;
				sb.append("\nContainer '" + LabelHelper.getFullPath(c, false) + "'");
			}
			else if( selection instanceof ContainerStartupOption ) {
				ContainerStartupOption c = (ContainerStartupOption)selection;
				sb.append("\nContainerStartupOption '" + c.getOptionName());
			}
			else if( selection instanceof Computer ) {
				Computer c = (Computer)selection;
				sb.append("\nComputer '" + LabelHelper.getComputerLabel(c) + "'");
			}
			else if( selection instanceof ChannelMapping ) {
				ChannelMapping dm = (ChannelMapping)selection;
				sb.append("\nChannel Mapping '" + LabelHelper.getChannelMappingLabel(dm));
			}
			else if( selection instanceof DomainsMapping ) {
				DomainsMapping dm = (DomainsMapping)selection;
				sb.append("\nDomain Mapping '" + LabelHelper.getDomainsMappingLabel(dm));
			}
		}

		return sb.toString();
	}


	private List<TmcdbObject> getEligibleElementsForDeletion(IStructuredSelection selection) {

		List<TmcdbObject> result = new ArrayList<TmcdbObject>();

		Iterator<?> it = selection.iterator();

		while(it.hasNext()) {

			Object o = it.next();

			if( o instanceof Computer ) {
				Computer c = (Computer)o;
				try {
					ComputerConversationUtils.getInstance().hydrateContainers(c);
				} catch (Exception e) {
					// TODO: exception handling
				}
				if( c.getContainers() == null || c.getContainers().size() == 0 )
					result.add(c);
				else
					MessageDialog.openInformation( _window.getShell(), "Cannot delete Computer",
							"Computer '" + LabelHelper.getComputerLabel(c) + "' cannot be deleted " +
					"because it still contains Containers");
			}
			else if( o instanceof Container) {
				Container c = (Container)o;
				try {
					ComponentConversationUtils.getInstance().hydrateComponents(c);
				} catch (Exception e) {
					// TODO: exception handling
				}
				if( c.getComponents() == null || c.getComponents().size() == 0 )
					result.add(c);
				else
					MessageDialog.openInformation( _window.getShell(), "Cannot delete Container",
							"Container '" + LabelHelper.getFullPath(c, false) +  "' cannot be deleted " +
					"because it still contains Components");
			}
			else if( o instanceof ContainerStartupOption ) {
				result.add((ContainerStartupOption) o);
			}
			else if( o instanceof AcsService ) {
				result.add((AcsService) o);
			}
			else if( o instanceof ChannelMapping ) {
				result.add((ChannelMapping) o);
			}
			else if( o instanceof DomainsMapping ) {
				result.add((DomainsMapping) o);
			}
			else if( o instanceof Component)
				result.add((Component)o);

			/* TODO: Check if there are monitoring points associated to this BACI property */
			else if( o instanceof BACIProperty)
				result.add((BACIProperty)o);

		}

		return result;
	}

	@Override
	public void selectionChanged(IWorkbenchPart part, ISelection selection) {

		if( selection instanceof IStructuredSelection && GuiUtils.isGodUser()) {

			_selection = (IStructuredSelection)selection;

			Iterator<?> it = _selection.iterator();
			Class<?> previousClazz = null;
			while(it.hasNext()) {
				Object o = it.next();
				if( !isClassAllowed(o) ) {
					setEnabled(false);
					return;
				}
				if( previousClazz != null && !o.getClass().equals(previousClazz) ) {
					setEnabled(false);
					return;
				}
				previousClazz = o.getClass();
			}
			setEnabled(true);
		}
		else
			setEnabled(false);

	}

	@Override
	public void dispose() {
		_window.getSelectionService().removeSelectionListener(this);
	}
	
	private boolean isClassAllowed(Object o) {
		return ( o instanceof AcsService ||
				 o instanceof Component ||
		         o instanceof Computer ||
		         o instanceof Container ||
		         o instanceof ContainerStartupOption ||
		         o instanceof ChannelMapping || 
				 o instanceof DomainsMapping || 
		         o instanceof BACIProperty );
	}

}
