/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.hibernate.exception.ConstraintViolationException;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.obops.tmcdbgui.views.ConfigurationsView;
import alma.tmcdb.domain.StartupScenario;

/**
 * Action to delete a startup scenario from a configuration. 
 * See COMP-4964 for the motivation for this action/use case.
 * 
 * @author sharring
 */
public class DeleteStartupScenarioAction extends ConversationalAction 
{
	private static final String ID = "delete_startupscenario.action";
	private IStructuredSelection selection;
	private StartupScenario startupScenario;
	private boolean cancelled;
	
	public DeleteStartupScenarioAction(IWorkbenchWindow window)
	{
		this.cancelled = false;
		this.window = window;
		window.getSelectionService().addSelectionListener(this);
		setId(ID);
		setText("Delete startup scenario");
		setToolTipText("Deletes a Startup Scenario");
		setImageDescriptor( RcpUtils.getImageDescriptor( "icons/startup-delete.png" ));
	}
	
	@Override public void doPreConversational() 
	{
		cancelled = !(MessageDialog.openConfirm(this.window.getShell(), "Please confirm", "Delete startup scenario: '" + startupScenario.getName() + "'?"));
	}
	
	@Override
	public void doConversational() 
	{
		if(cancelled) {
			return;
		}
		try {
			window.getShell().setCursor(window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
		    HwConfigurationConversationUtils.getInstance().removeEntireStartupScenario(startupScenario, true);
		} 
		catch(ConstraintViolationException ex) 
		{
			MessageDialog.openWarning(window.getShell(), "Cannot delete startupscenario", "Constraints exist which prevent deletion" );
		}
		catch (Exception e) 
		{
			MessageDialog.openError(window.getShell(), "Cannot delete startupscenario", "This startupscenario cannot be deleted for unknown reasons." );
			e.printStackTrace();
		}
		finally {
			window.getShell().setCursor(null);
		}
	}

	@Override
	public void doPostConversational() {
		if(cancelled) {
			return;
		}
		Shell shell = window.getShell();
		try {
			shell.setCursor(shell.getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			ConfigurationsView view = (ConfigurationsView) RcpUtils.findView(ConfigurationsView.ID);
			view.externalModelChange();
		}
		finally {
			shell.setCursor(null);
		}
	}

	@Override
	public void selectionChanged(IWorkbenchPart workbenchPart, ISelection newSelection) 
	{
		if( newSelection instanceof IStructuredSelection ) 
		{
			selection = (IStructuredSelection)newSelection;
			if( selection.size() == 1 && (selection.getFirstElement() instanceof StartupScenario))
			{
				this.startupScenario = (StartupScenario) selection.getFirstElement();
				setEnabled(true);
			}
			else 
			{
				setEnabled(false);
			}
		}
		else {
			setEnabled(false);
		}
	}

}
