/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * 
 */
package alma.obops.tmcdbgui.handlers;

import java.lang.reflect.InvocationTargetException;
import java.util.List;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.ActionFactory.IWorkbenchAction;
import org.hibernate.criterion.MatchMode;

import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.Configuration;
import alma.archive.tmcdb.persistence.ComponentNameHelper;
import alma.obops.dam.ServiceException;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.dialogs.ConfigurationSelectionDialog;
import alma.obops.tmcdbgui.dialogs.ConfigurationSelectionDialogLabelProvider;
import alma.obops.tmcdbgui.utils.conversation.ComponentConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.obops.tmcdbgui.views.SoftwareDeploymentView;
import alma.tmcdb.domain.HwConfiguration;

/**
 * Copies a component, together with its BACI properties, to the
 * necessary configuration.
 *
 * @author rtobar, Apr 28th, 2011
 *
 */
public class CopyComponentAction extends Action implements ISelectionListener, IWorkbenchAction {

	static public String ID = "alma.obops.tmcdbgui.";

	public enum CopyOperation {
		COPY,
		CLONE
	}

	private CopyOperation _operation;
	private Component _component;
	private IWorkbenchWindow _window;
	private SoftwareDeploymentView _sdv;

	public CopyComponentAction(CopyOperation operation, IWorkbenchWindow window, SoftwareDeploymentView sdv) {
		_operation = operation;
		_window = window;
		_sdv = sdv;
		_window.getSelectionService().addSelectionListener(this);
		setId(ID);
		setText( (CopyOperation.CLONE.equals(operation) ? "&Clone" : "&Copy") + " component");
		setToolTipText( (CopyOperation.CLONE.equals(operation) ?
		   "Clones a component inside the same configuration" :
		   "Copies a component to a different configuration"));
		setImageDescriptor( RcpUtils.getImageDescriptor( "icons/component-" + (CopyOperation.CLONE.equals(operation) ? "clone.png" : "copy.png")) );
	}

	@Override
	public void run() {

		Configuration targetConfiguration = null;
		String newName = null;
		String newPath = null;

		if( _operation.equals(CopyOperation.CLONE) ) {

			InputDialog dialog = new InputDialog(_window.getShell(), "New component name", "Insert the new full path+name for the cloned component",
			                      ComponentNameHelper.getFullName( _component.getPath(), _component.getComponentName()),
			                      new IInputValidator() {
									private String regexp = "[0-9a-zA-Z_-]+(/[0-9a-zA-Z_-]+)*";
									@Override
									public String isValid(String newText) {
										if( newText.matches(regexp) )
											return null;
										return "Component path/name invalid";
									}
								});
			if( dialog.open() == Dialog.CANCEL )
				return;

			String value = dialog.getValue();
			if( value == null || value.trim().length() == 0 ) {
				RcpUtils.infoMessage(_window.getShell(), "No new name given", "No name was given for the new component. No action will be taken.");
				return;
			}

			String []tmp = ComponentNameHelper.getPathAndName(value);
			targetConfiguration = _component.getConfiguration();
			String adjustedPath = (tmp[0] == null || tmp[0].equals("")) ? "/" : tmp[0];
			newPath = adjustedPath;
			newName = tmp[1];
		}

		else if ( _operation.equals(CopyOperation.COPY) ) {
			
			ConfigurationSelectionDialog dialog = new ConfigurationSelectionDialog(_window.getShell(), new ConfigurationSelectionDialogLabelProvider());
			if( dialog.open() == Dialog.CANCEL )
				return;

			Object configNames[] = dialog.getResult();
			if( configNames == null || configNames.length != 1) {
				RcpUtils.infoMessage(_window.getShell(), "No Configuration selected", "No configuration was selected. No action will be taken");
				return;
			}
			
			List<HwConfiguration> matchingHwConfigs = null;
			try {
				matchingHwConfigs = HwConfigurationConversationUtils.getInstance().findConfigurationsByName((String) configNames[0], MatchMode.EXACT);
			} catch (Exception e) {
				matchingHwConfigs = null;
			}
			
			if(matchingHwConfigs != null && matchingHwConfigs.size() == 1) {
				targetConfiguration = matchingHwConfigs.get(0).getSwConfiguration();
			}
			else {
				RcpUtils.infoMessage(_window.getShell(), "Error", "Problem loading chosen configuration. No action will be taken");
				return;
			}
			
			if( targetConfiguration.getConfigurationName().equals(_component.getConfiguration().getConfigurationName()) ) {
				RcpUtils.infoMessage(_window.getShell(), "Invalid Configuration",
				   "The selected configuration is the same that the origin configuration, copy will not take place.");
				return;
			}
				
			newPath = _component.getPath();
			newName = _component.getComponentName();
		}

		try {
			ComponentConversationUtils.getInstance().cloneAndStoreComponent(_component, targetConfiguration, newName, newPath);
		} 
		catch(InvocationTargetException e) {
			if(e.getTargetException() instanceof ServiceException) {
				RcpUtils.errorMessage((ServiceException)e.getTargetException(), _window.getShell(), "Error while copying/cloning component",
						e.getTargetException().getMessage());
			} else {
				RcpUtils.errorMessage(e, _window.getShell(), "Error while copying/cloning component", "There was an error while copying/cloning your component. Please check the details and refer to the developers.");
			}
		}
		catch(Exception e) {
			RcpUtils.errorMessage(e, _window.getShell(), "Error while copying/cloning component", "There was an error while copying/cloning your component. Please check the details and refer to the developers.");
		}

		// Refresh the view
		_sdv.internalModelChange();
	}

	@Override
	public void dispose() {
		_window.getSelectionService().removeSelectionListener(this);
	}

	@Override
	public void selectionChanged(IWorkbenchPart part, ISelection selection) {
		if( selection instanceof IStructuredSelection ) {
			IStructuredSelection ssel = (IStructuredSelection)selection;
			if( ssel.getFirstElement() instanceof Component ) {
				_component = (Component)ssel.getFirstElement();
				setEnabled(true);
				return;
			}
		}
		setEnabled(false);
	}

}
