/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import java.lang.reflect.InvocationTargetException;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchWindow;

import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.conversation.BaseElementConversationUtils;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.HwConfiguration;

/**
 * Abstract class which can be used to copy base elements of various flavors (via subclassing,
 * and providing concrete implementations of two small abstract methods). At present, we only
 * have a use case to copy antennas, but we've abstracted things out such that we *could* copy
 * any base element via simple extensions of this class. The 'reference implementation' is
 * the CopyAntennaAction; see it for an example extension.
 *
 * @author sharring
 * @see CopyAntennaAction
 */
public abstract class CopyBaseElementAction extends ConversationalAction
{
	protected static final String COPYING_FAILED_TITLE = "Copying failed";
	protected static final String COPY_BASE_ELEMENT = "Copy";
	protected BaseElement baseElement;
	protected BaseElement existingBaseElement;
	protected String name;
	protected boolean cancelled = false;

	protected HwConfiguration addToConfiguration;
	protected IWorkbenchWindow _window;

	public CopyBaseElementAction(IWorkbenchWindow win)
	{
		_window = win;
    	_window.getSelectionService().addSelectionListener(this);
    	setText( COPY_BASE_ELEMENT );
	}

	@Override public void doPreConversational()
	{
		Shell shell = _window.getShell();
		try {
			shell.setCursor(shell.getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			// Collect user input
			cancelled = false;
			Wizard wizard = createWizard();
			WizardDialog dialog = new WizardDialog( shell, wizard );
			int ret = dialog.open();
			if( ret == WizardDialog.CANCEL ) {
				cancelled = true;
				return;
			}
		}
		catch( Exception e ) {
			e.printStackTrace();
			MessageDialog.openError( shell,
					"Copying of Base Element ",
					e.getClass().getSimpleName() +
					": " + e.getMessage() );
		}
		finally {
			shell.setCursor(null);
		}
	}

	/**
	 * Subclasses must implement this to create the wizard of their choosing.
	 * @return the wizard to use for the copy action.
	 */
	protected abstract Wizard createWizard();

	@Override
	public void doConversational()
	{
		Shell shell = _window.getShell();

		if(!cancelled)
		{
			try {
				shell.setCursor(shell.getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
				doCopy();
			}
			catch( Exception e ) {
				String message = "Unexpected exception during cloning of base element.";

				MessageDialog.openError( shell,
						COPYING_FAILED_TITLE,
						message );
				e.printStackTrace();
			}
			finally {
				shell.setCursor(null);
			}
		}
	}

	@Override
	public void doPostConversational()
	{
		if(!cancelled)
		{
			Shell shell = _window.getShell();
			try {
				shell.setCursor(shell.getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
				HwConfiguration[] configs = new HwConfiguration[1];
				configs[0] = addToConfiguration;
				//view.expandToLevel(1);
				this.modelShouldBeReloaded();
			}
			finally {
				shell.setCursor(null);
			}
		}
	}

	/**
	 * Setter for the name of the newly copied antenna.
	 * @param copiedAntennaName the name of the newly copied antenna.
	 */
	public void setName(String copiedAntennaName) {
		this.name = copiedAntennaName;
	}

	/**
	 * Setter for the configuration to which the antenna will be copied.
	 * @param configuration the configuration to which the antenna will be copied.
	 */
	public void setAddToConfiguration(HwConfiguration configuration) {
		this.addToConfiguration = configuration;
	}

	/**
	 * Default implementation of the copy, which can be overridden to do the actual copying, if needed.
	 * Typical future subclasses may not need to override this default behavior. In the case of our
	 * 'reference implementation' of a subclass, CopyAntennaAction, it was necessary to override this
	 * method to assign the antenna's component. Typically, that may not be needed for other base elements,
	 * should we ever copy them via subclassing this abstract class, so we provide a default implementation.
	 *
	 * @throws NoSuchMethodException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 */
	protected void doCopy()
	   throws Exception
	{
		BaseElementConversationUtils.getInstance().copyBaseElement(baseElement, name, addToConfiguration);
	}

	@Override
	public void dispose() {
		_window.getSelectionService().removeSelectionListener(this);
	}
}
