/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.ActionFactory.IWorkbenchAction;

import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.handlers.conversation.ModelPublisherAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.views.ConfigurationsView;
import alma.tmcdb.domain.AntennaToPad;

/**
 * Closes out an antenna to pad assignment.
 * @author sharring
 */
public class CloseAntennaToPadAssignmentAction extends ModelPublisherAction implements
		ISelectionListener, IWorkbenchAction 
{
	private IStructuredSelection selection;
	private IWorkbenchWindow window;
	private String ID = "alma.obops.tmcdbgui.handlers.CloseAntennaToPadAssignmentAction";
	
	public CloseAntennaToPadAssignmentAction(IWorkbenchWindow window, IModelChangeListener listener) 
    {
    	this.window = window;
    	setId(ID);
        setText("Unassign");
		setToolTipText("Closes an antenna to pad assignment");
    	window.getSelectionService().addSelectionListener(this);
    	this.addModelChangeListener(listener);
    }
	
	@Override
	public void selectionChanged(IWorkbenchPart win, ISelection newSelection) {
		if( newSelection instanceof IStructuredSelection && GuiUtils.isGodUser()) {
			selection = (IStructuredSelection)newSelection;
			setEnabled( selection.size() == 1 &&
			            selection.getFirstElement() instanceof AntennaToPad);	
		}
	}

	@Override
	public void dispose() {
		window.getSelectionService().removeSelectionListener(this);
	}
	
	public void run() 
	{
		AntennaToPad a2p = (AntennaToPad)selection.getFirstElement();
		a2p.setEndTime(System.currentTimeMillis());
		try {
			window.getShell().setCursor(window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			AssignAntennaToPadAction.closeOutAntToPad(a2p, ConversationToken.CONVERSATION_COMPLETED);
			this.modelChanged();
			ConfigurationsView configView = (ConfigurationsView)RcpUtils.findView(ConfigurationsView.ID);
			configView.getConfigurationsTreeViewer().refresh();
		} catch (Exception e) {
			throw new RuntimeException("Could not update antenna to pad assignment", e);
		} finally {
			window.getShell().setCursor(null);
		}
	}
}
