/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;


import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.StartupScenarioConversationUtils;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.StartupScenario;

/**
 * Action to clone a startup scenario.
 * @author sharrington
 */
public class CloneStartupScenarioAction extends ConversationalAction 
{
	private static final String ID = "clone_startup.action";
	private static final String CLONING_FAILED_TITLE = "Cloning failed";
	protected StartupScenario scenario;
	private String name;
	private StartupScenario existingScenario;
	private boolean cancelled;

	/**
     * Constructor.
     * @param window the window associated with this action.
     */
	public CloneStartupScenarioAction( IWorkbenchWindow window ) 
	{
		this.window = window;
		setText( "Clone Startup Scenario" );
		setId(ID);
		setImageDescriptor( RcpUtils.getImageDescriptor( "icons/startup.png" ));
	}

	/**
	 * Setter for the startup scenario that is being cloned.
	 * @param scenario the startup scenario that is to be cloned.
	 */
	public void setScenario(StartupScenario scenario) 
	{
		this.scenario = scenario;
	}
	
	/**
	 * @return the name
	 */
	public String getName() {
		return name;
	}
	
	@Override public void doPreConversational()
	{
		Shell shell = window.getShell();

		// prompt for the name of the cloned startup scenario
		InputDialog inputDialog = new InputDialog(shell, "Specify name", 
				"Please specify the name of the cloned startup scenario.", 
				"Copy of: " + scenario.getName(), null);

		if(inputDialog.open() != Window.CANCEL)
		{
			cancelled = false;
			name = inputDialog.getValue();
			try {
				shell.setCursor(new Cursor(shell.getDisplay(), SWT.CURSOR_WAIT));
				existingScenario = findExistingScenario(scenario.getConfiguration(), name);
			} catch (Exception th) {
				th.printStackTrace();
				throw new RuntimeException("Could not query existing startup scenarios.", th);
			} finally {
				shell.setCursor(null);
			}
		} else {
			cancelled = true;
		}
	}

	private StartupScenario findExistingScenario(HwConfiguration confToSearch, String name2find) 
	{
		StartupScenario retVal = null;
		for(StartupScenario scenario1 : confToSearch.getStartupScenarios())
		{
			if(scenario1.getName().equals(name2find)) {
				retVal = scenario1;
				break;
			}
		}
		return retVal;
	}

	@Override
	public void doConversational() 
	{		
		if(cancelled) {
			return;
		}
		
		Shell shell = window.getShell();
		
		if(null == existingScenario)
		{
			try {	
				shell.setCursor(new Cursor(shell.getDisplay(), SWT.CURSOR_WAIT));
				StartupScenarioConversationUtils.getInstance().cloneStartupScenario(scenario, name);
			} 
			catch( Exception e ) {
				String message = "Unexpected exception during cloning of startup scenario.";
				
				MessageDialog.openError( shell, 
						CLONING_FAILED_TITLE,
						message );
				e.printStackTrace();
			} finally {
				shell.setCursor(null);
			}
		} 
		else {
			MessageDialog.openWarning(shell, "Uniqueness constraint violated", 
					"A startup scenario with the name: '" + name + "' already exists in the configuration.");
		}
	}

	@Override
	public void doPostConversational() 
	{
		if(cancelled) {
			return;
		}
		
		scenario.getConfiguration();

		Shell shell = window.getShell();
		
		try {
			shell.setCursor(new Cursor(shell.getDisplay(), SWT.CURSOR_WAIT));
			this.modelShouldBeReloaded();		
		} finally {
			shell.setCursor(null);
		}
	}
	
	public void setName( String name ) {
		this.name = name;
	}

	@Override
	public void selectionChanged(IWorkbenchPart part, ISelection newSelection) 
	{
		if( newSelection instanceof IStructuredSelection && GuiUtils.isGodUser()) 
		{
			selection = (IStructuredSelection)newSelection;
			if( selection.size() == 1 &&
					   (selection.getFirstElement() instanceof StartupScenario) )	
			{
				setEnabled(true);
				this.scenario = (StartupScenario) selection.getFirstElement();
			}
			else {
				setEnabled(false);
			}
		}
		else {
			setEnabled(false);
		}
	}
}
