/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import java.lang.reflect.InvocationTargetException;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.hibernate.criterion.MatchMode;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.tmcdb.domain.HwConfiguration;

/**
 * Action to clone a configuration.
 * @author sharrington
 */
public class CloneConfigurationAction extends ConversationalAction
{
	private String name;
	private List<HwConfiguration> existingConfigs;
	private String newName;
	private boolean cancelled;

	private String ID = "clone_configuration.action";
    private IWorkbenchWindow _window;

	/** 
	 * Public constructor 
	 */
	public CloneConfigurationAction(IWorkbenchWindow window, IModelChangeListener listener) {
		_window = window;
		setId(ID);
		setText("Clone Configuration");
		setToolTipText("Clones an existing Configuration");
		setImageDescriptor( RcpUtils.getImageDescriptor( "icons/configuration-clone.png" ));
    	_window.getSelectionService().addSelectionListener(this);
    	this.addModelChangeListener(listener);
	}

	/**
	 * @return the name
	 */
	public String getName() {
		return name;
	}

	@Override 
	public void doPreConversational()
	{
		Shell shell = _window.getShell();

		// prompt for the name of the cloned configuration
		InputDialog inputDialog = new InputDialog(shell, 
				"Specify name", 
				"Please specify the name of the cloned configuration.", 
				"Copy of: " + ((HwConfiguration)selection.getFirstElement()).getName(), null);

		if(inputDialog.open() != Window.CANCEL)
		{			
			cancelled = false;
			shell.setCursor(shell.getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			newName = inputDialog.getValue();			

			try {
				existingConfigs =  HwConfigurationConversationUtils.getInstance().findConfigurationsByName(newName, MatchMode.EXACT);
			} catch(Exception th) {
				existingConfigs = null;
				MessageDialog.openError( shell, "Query failed", th.getMessage() );
				th.printStackTrace();
			}
			finally {
				shell.setCursor(null);
			}
		} else {
			cancelled = true;
			existingConfigs = null;
		}
	}
	
	@Override
	public void doConversational() 
	{
		if(cancelled) {
			return;
		}
		
		Shell shell = _window.getShell();
		if(existingConfigs != null && existingConfigs.size() == 0) 
		{
			try {
				shell.setCursor(shell.getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
				new ProgressMonitorDialog(shell).
				   run(true, false, new LongRunningOperation());
				
				name = newName;
			}
			catch( Exception e ) {
				e.printStackTrace();
				String message = "Unexpected exception during cloning of configuration.";
				MessageDialog.openError( shell, "Cloning failed", message );
			} 
			finally {
				shell.setCursor(null);
			}
		} else {
			MessageDialog.openWarning(shell, "Uniqueness constraint violated", 
					"A configuration with the name: '" + newName + "' already exists.");
		}
	} 

	@Override
	public void doPostConversational() 
	{	
		if(!cancelled) {
			this.modelShouldBeReloaded();
		}
	}


	public void setName( String name ) {
		this.name = name;
	}

	@Override
	public void selectionChanged(IWorkbenchPart part, ISelection newSelection) {
		if( newSelection instanceof IStructuredSelection && GuiUtils.isGodUser()) {
			selection = (IStructuredSelection)newSelection;
			setEnabled( selection.size() == 1 && 
			            selection.getFirstElement() instanceof HwConfiguration );
		}
		else {
			setEnabled(false);
		}
	}

	@Override
	public void dispose() {
		_window.getSelectionService().removeSelectionListener(this);
	}
	
	private class LongRunningOperation implements IRunnableWithProgress {

		  /**
		   * LongRunningOperation constructor
		   * 
		   * @param indeterminate whether the animation is unknown
		   */
		  public LongRunningOperation() {
		  }

		  /**
		   * Runs the long running operation
		   * 
		   * @param monitor the progress monitor
		   */
		  public void run(IProgressMonitor monitor) throws InvocationTargetException,
		      InterruptedException 
		   {
		    monitor.beginTask("Cloning configuration - this is a long running operation.", IProgressMonitor.UNKNOWN);
		    try {
		    	HwConfigurationConversationUtils.getInstance().cloneConfiguration((HwConfiguration)selection.getFirstElement(), newName);	
		    }
		    catch(Exception ex) {
		    	throw new RuntimeException("Could not complete cloning", ex);
		    }
		    monitor.done();
		  }
	}

}
