/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;

import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.conversation.BaseElementConversationUtils;
import alma.tmcdb.domain.BaseElement;

/**
 * Abstract base class used for common logic for cloning base elements.
 * @author sharring
 */
public abstract class CloneBaseElementAction extends ConversationalAction
{
	protected static final String CLONE_BASE_ELEMENT = "Clone";
	protected static final String CLONING_FAILED_TITLE = "Cloning failed";
	protected Cursor oldCursor;
	protected String name;
	protected boolean cancelled = false;
	protected IWorkbenchWindow workbenchWindow;

	/**
	 * Constructor
	 * @param baseElement the base element to clone
	 * @param view the TreeViewer associated with the action.
	 */
	public CloneBaseElementAction(IWorkbenchWindow window) 
	{
		workbenchWindow = window;
		workbenchWindow.getSelectionService().addSelectionListener(this);
	}
	
	@Override
	public void doConversational() 
	{				
		if(!cancelled)
		{
			try {	
				workbenchWindow.getShell().setCursor(new Cursor(workbenchWindow.getShell().getDisplay(), SWT.CURSOR_WAIT));
				BaseElementConversationUtils.getInstance().cloneBaseElement((BaseElement)selection.getFirstElement(), name);
			} 
			catch( Exception e ) {
				String message = "Unexpected exception during cloning of base element.";

				MessageDialog.openError( workbenchWindow.getShell(), 
						CLONING_FAILED_TITLE,
						message );
				e.printStackTrace();
			}
			finally {
				workbenchWindow.getShell().setCursor(null);
			}
		}
	} 

	@Override
	public void doPostConversational() {

		Shell shell = workbenchWindow.getShell();
		shell.setCursor(shell.getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
		if(!cancelled)
		{
			this.modelShouldBeReloaded();
		}
		shell.setCursor(null);
	}

	/**
	 * Setter for the name for the cloned base element.
	 * @param name the name of the cloned base element.
	 */
	public void setName(String name)
	{
		this.name = name;
	}

	/**
	 * Selection listener. Subclasses must make sure that the first element of the selection is of type
	 * {@link BaseElement}.
	 * 
	 * @see org.eclipse.ui.ISelectionListener#selectionChanged(org.eclipse.ui.IWorkbenchPart, org.eclipse.jface.viewers.ISelection)
	 */
	public abstract void selectionChanged(IWorkbenchPart part, ISelection theSelection);
}
