/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import java.util.HashMap;
import java.util.Map;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.ActionFactory.IWorkbenchAction;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.handlers.conversation.ModelPublisherAction;
import alma.obops.tmcdbgui.utils.conversation.BaseElementConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.obops.tmcdbgui.widgets.ScrollableDialog;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.HwConfiguration;

public class CheckForDuplicatedLoOffsetsAction extends ModelPublisherAction implements ISelectionListener, IWorkbenchAction 
{
	private HwConfiguration configuration;
	private IStructuredSelection selection;
	private IWorkbenchWindow window;
	private String ID = "alma.obops.tmcdbgui.handlers.CheckForDuplicatedLoOffsetsAction";
	
	public CheckForDuplicatedLoOffsetsAction(IWorkbenchWindow window, IModelChangeListener listener) 
    {
    	this.window = window;
    	setId(ID);
    	this.setImageDescriptor(RcpUtils.getImageDescriptor("icons/check-duplicates.gif"));
        setText("Check for duplicate LO offsets");
		setToolTipText("Checks for antennas with the same LO offset");
    	window.getSelectionService().addSelectionListener(this);
    	this.addModelChangeListener(listener);
    }
	
	@Override
	public void selectionChanged(IWorkbenchPart part, ISelection newSelection) 
	{
		if( newSelection instanceof IStructuredSelection) 
		{
			selection = (IStructuredSelection)newSelection;
			
			if(selection.getFirstElement() instanceof HwConfiguration) {
				this.configuration = (HwConfiguration) selection.getFirstElement();	
				setEnabled( selection.size() == 1);
			}
		}
	}

	@Override
	public void dispose() {
		window.getSelectionService().removeSelectionListener(this);
	}
	
	@Override
	public void run() 
	{
		try 
		{
			window.getShell().setCursor(window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			try {
				HwConfigurationConversationUtils.getInstance().hydrateBaseElements(configuration);
			} catch (Exception e1) {
				e1.printStackTrace();
				MessageDialog.openError(window.getShell(), "Problem encountered", "Could not hydrate base elements");
				return;
			}

			Map<Integer, StringBuffer> LOToBufferMap = new HashMap<Integer, StringBuffer>();
			boolean hasDuplicates = false;
			for(BaseElement be: configuration.getBaseElements()) 
			{
				if(be.getType().equals(BaseElementType.Antenna) && be instanceof Antenna) 
				{
					Antenna antenna = (Antenna) be;
					Antenna[] totalAntennasWithLO;
					try {
						BaseElementConversationUtils.getInstance().hydrateAntenna(antenna);
						totalAntennasWithLO = HwConfigurationConversationUtils.getInstance().findAntennaByLoOffsetInConfig(antenna.getLoOffsettingIndex(), configuration);
					} catch (Exception e) {
						e.printStackTrace();
						MessageDialog.openError(window.getShell(), "Problem encountered", "Could not query antennas/LO offsets");
						return;
					}
					if(totalAntennasWithLO.length > 1) 
					{
						StringBuffer duplicates = LOToBufferMap.get(antenna.getLoOffsettingIndex());
						if(null == duplicates) 
						{
							duplicates = new StringBuffer();
							hasDuplicates = true;
							duplicates.append("These antennas have the LO offset value of '").append(antenna.getLoOffsettingIndex().toString()).append("':\n");
							int count = 0;
							for(Antenna antIterated : totalAntennasWithLO) 
							{
								duplicates.append(antIterated.getName());
								if(++count < totalAntennasWithLO.length) {
									duplicates.append(", ");
									int remainder = count % 12;
									if(remainder == 0) {
										duplicates.append("\n");
									}
								} 
							}
							LOToBufferMap.put(antenna.getLoOffsettingIndex(), duplicates);
						}
					}
				}
			}
			
			if(hasDuplicates) 
			{
				StringBuffer strBuf = new StringBuffer();
				for(Map.Entry<Integer, StringBuffer> entry: LOToBufferMap.entrySet()) 
				{
					strBuf.append(entry.getValue().toString()).append("\n\n");
				}
				ScrollableDialog scrollDialog = new ScrollableDialog(window.getShell(), "Duplicates found", "The following duplicates were found:", strBuf.toString());
				scrollDialog.create();
				scrollDialog.open();
			} else {
				MessageDialog.openInformation(window.getShell(), "No duplicates found", "There were no duplicated LO sequence values found");
			}
		}
		finally {
			window.getShell().setCursor(null);
		}
	}

}
