/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import java.util.HashSet;
import java.util.Set;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;

import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.BackendConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.BaseElementConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.HolographyTowerToPadConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.obops.tmcdbgui.views.ConfigurationsView;
import alma.obops.tmcdbgui.views.providers.helpers.config.PadHelper;
import alma.obops.tmcdbgui.wizards.ChooseHolographyTowerWizard;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.HolographyTower;
import alma.tmcdb.domain.HolographyTowerToPad;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.Pad;

public class AssignPadToHolographyTowerAction extends ConversationalAction 
{
	private String ID = "assign_pad_to_holographytower.action";
    private IWorkbenchWindow _window;
    private Pad pad;
    
	/** 
	 * Public constructor 
	 */
	public AssignPadToHolographyTowerAction(IWorkbenchWindow window, IModelChangeListener listener)
	{
		_window = window;
		setId(ID);
		setText("Assign pad to holography tower");
		setToolTipText("Assigns a pad to a holography tower");
		setImageDescriptor( RcpUtils.getImageDescriptor( "icons/padtoholographytower.png" ));
    	_window.getSelectionService().addSelectionListener(this);
    	this.addModelChangeListener(listener);
	}
	
	public static void addPadToHolographyTower(Pad padToAdd, HolographyTower assignedHolographyTower)
	{
		try 
		{
			BaseElementConversationUtils.getInstance().hydrateHolographyTower(assignedHolographyTower);
			
			// close out the old holographytower assignment, if any, for the dragged pad
			HolographyTowerToPad currentHolographyTowerToPad = PadHelper.findCurrentHolographyTowerToPadForPad(padToAdd);
			
			if(null != currentHolographyTowerToPad) 
			{
				if(!currentHolographyTowerToPad.getPad().getId().equals(assignedHolographyTower.getId())) 
				{
					BaseElementConversationUtils.getInstance().hydratePad(currentHolographyTowerToPad.getPad());
					HolographyTower ht = findHolographyTowerInConfiguration(padToAdd.getConfiguration(), currentHolographyTowerToPad.getHolographyTower());
					if(ht == null && currentHolographyTowerToPad.getHolographyTower().getConfiguration().getId().equals(padToAdd.getConfiguration().getId()) ) 
					{
						HwConfigurationConversationUtils.getInstance().hydrateBaseElements(currentHolographyTowerToPad.getHolographyTower().getConfiguration());
						ht = findHolographyTowerInConfiguration(currentHolographyTowerToPad.getHolographyTower().getConfiguration(), currentHolographyTowerToPad.getHolographyTower());
					}
					if(ht != null) 
					{
						BaseElementConversationUtils.getInstance().hydrateHolographyTower(ht);
						boolean removed = removeH2pFromHolographyTower(ht, currentHolographyTowerToPad);
						if(!removed) {
							Thread.dumpStack();
							throw new RuntimeException("Item was not removed from h2p mappings...");
						}
					}
				}
			}
			
			// create new h2p assignment & persist it
			HolographyTowerToPad h2p = new HolographyTowerToPad(assignedHolographyTower, padToAdd);
			h2p.setAzimuth(0d);
			h2p.setElevation(0d);
			HolographyTowerToPadConversationUtils.getInstance().saveOrUpdateHolographyTowerToPad(h2p);
			BaseElementConversationUtils.getInstance().saveOrUpdateHolographyTower(assignedHolographyTower);

			// notify any interested parties
			ConfigurationsView view = (ConfigurationsView)RcpUtils.findView(ConfigurationsView.ID);
			view.internalModelChange();
			
			// extra refresh to make sure we don't have an expanded tree node that has no children (due to deletion/unassignment)
			view.getConfigurationsTreeViewer().refresh();
		}
		catch( Exception e ) {
			throw new RuntimeException("Problem assigning HolographyTower to Pad.", e);
		}
	}
	
	private static HolographyTower findHolographyTowerInConfiguration(
			HwConfiguration configuration, HolographyTower holographyTower) throws Exception 
	{
		HolographyTower retVal = null;
		
		for(BaseElement be: configuration.getBaseElements()) 
		{
			if(be.getType().equals(BaseElementType.HolographyTower) && be instanceof HolographyTower)
			{
				HolographyTower iteratedHolographyTower = (HolographyTower)be;
				BaseElementConversationUtils.getInstance().hydrateHolographyTower(iteratedHolographyTower);
				if(iteratedHolographyTower.getId().equals(holographyTower.getId())) {
					retVal = iteratedHolographyTower;
					break;
				}
			}
		}
		
		return retVal;
	}

	public static boolean removeH2pFromHolographyTower(HolographyTower holographyTower, HolographyTowerToPad currentHolographyTowerToPad) throws Exception 
	{
		boolean retVal = false;
		
		Set<HolographyTowerToPad> doNotDeleteH2ps = new HashSet<HolographyTowerToPad>();
		HolographyTowerToPad deleteH2p = null;
		for(HolographyTowerToPad h2p : holographyTower.getAssociatedPads()) {
			if(!h2p.getHolographyTower().getId().equals(currentHolographyTowerToPad.getHolographyTower().getId()) ||  
			   !h2p.getPad().getId().equals(currentHolographyTowerToPad.getPad().getId()))
			{
				doNotDeleteH2ps.add(h2p);
			} 
			else {
				deleteH2p = h2p;
			}
		}
		
		if(null != deleteH2p) {
			BackendConversationUtils.getInstance().delete(deleteH2p, ConversationToken.CONVERSATION_PENDING);
			retVal = true;
		}
		holographyTower.setAssociatedPads(doNotDeleteH2ps);
		BaseElementConversationUtils.getInstance().saveOrUpdateHolographyTower(holographyTower);
		return retVal;
	}

	@Override
	public void doConversational() 
	{
		Shell shell = _window.getShell();

		try {
			// set a wait cursor
			_window.getShell().setCursor(_window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			
			// Collect user input
			ChooseHolographyTowerWizard wizard = new ChooseHolographyTowerWizard( pad.getConfiguration() );
			WizardDialog dialog = new WizardDialog( shell, wizard );
			int ret = dialog.open();
			if( ret == WizardDialog.CANCEL ) {
				return;
			}
			
			// set a wait cursor
			_window.getShell().setCursor(_window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));

			HolographyTower chosenHolographyTower = wizard.getHolographyTower();
			
			// Create and save an antennaToPad assignment with the given info
			AssignPadToHolographyTowerAction.addPadToHolographyTower(pad, chosenHolographyTower);
		}
		catch( Exception e ) {
			e.printStackTrace();
			throw new RuntimeException("Could not create AntennaToPad assignment", e);
		} finally {
			// remove the wait cursor
			_window.getShell().setCursor(null);
		}
	}

	@Override
	public void doPostConversational() {
		// noop as it was already done in the static method
	}

	@Override
	public void selectionChanged(IWorkbenchPart wbpart, ISelection newSelection) 
	{
		if( newSelection instanceof IStructuredSelection && GuiUtils.isGodUser()) 
		{
			selection = (IStructuredSelection)newSelection;
			setEnabled( selection.size() == 1 && 
			            selection.getFirstElement() instanceof Pad );
			if(isEnabled())
			{
				this.pad = (Pad) selection.getFirstElement();
			}
		}
		else {
			setEnabled(false);
		}
	}
	
	@Override
	public void dispose() {
		window.getSelectionService().removeSelectionListener(this);
	}

}
