/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;

import alma.acs.util.UTCUtility;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AntennaToPadConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.BaseElementConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.obops.tmcdbgui.views.ConfigurationsView;
import alma.obops.tmcdbgui.views.providers.helpers.config.AntennaHelper;
import alma.obops.tmcdbgui.wizards.ChoosePadWizard;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.AntennaToPad;
import alma.tmcdb.domain.Pad;

public class AssignAntennaToPadAction extends ConversationalAction 
{
	private String ID = "assign_antenna_to_pad.action";
    private IWorkbenchWindow _window;
    private Antenna antenna;
    
	/** 
	 * Public constructor 
	 */
	public AssignAntennaToPadAction(IWorkbenchWindow window, IModelChangeListener listener)
	{
		_window = window;
		setId(ID);
		setText("Assign antenna to pad");
		setToolTipText("Assigns an antenna to a pad");
		setImageDescriptor( RcpUtils.getImageDescriptor( "icons/antennatopad.png" ));
    	_window.getSelectionService().addSelectionListener(this);
    	this.addModelChangeListener(listener);
	}
	
	public static void addAntennaToPad(Antenna antennaToAdd, Pad assignedPad)
	{
		try 
		{
			BaseElementConversationUtils.getInstance().hydratePad(assignedPad);
			
			if(!antennaToAdd.getConfiguration().getId().equals(assignedPad.getConfiguration().getId())) {
				HwConfigurationConversationUtils.getInstance().hydrateConfigurationHashCode(antennaToAdd.getConfiguration().getGlobalConfiguration());
			}
			
			// close out the old antennaToPad assignment, if any, for the antenna in our local configuration
			AntennaToPad currentAntToPad = AntennaHelper.findCurrentAntennaToPadForAntenna(antennaToAdd);
			if(null != currentAntToPad) 
			{
				if(!currentAntToPad.getPad().getId().equals(assignedPad.getId())) 
				{
					BaseElementConversationUtils.getInstance().hydratePad(currentAntToPad.getPad());
					closeOutAntToPad(currentAntToPad, ConversationToken.CONVERSATION_COMPLETED);
				}
			}
			
			// close out the old antennaToPad assignment, if any, for the antenna in our global configuration
			if(null != antennaToAdd.getConfiguration().getGlobalConfiguration()) {
				AntennaToPad currentAntToPadGlobal = BaseElementConversationUtils.getInstance().findOpenAntennaToPadAssignmentForAntennaInGlobal(antennaToAdd);
				if(null != currentAntToPadGlobal) 
				{
					if(!currentAntToPadGlobal.getPad().getName().equals(assignedPad.getName())) 
					{
						BaseElementConversationUtils.getInstance().hydratePad(currentAntToPadGlobal.getPad());
						closeOutAntToPad(currentAntToPadGlobal, ConversationToken.CONVERSATION_COMPLETED);
					}
				}
			}
			
			// create new a2p assignment & persist it
			Long currentTime = UTCUtility.utcJavaToOmg(System.currentTimeMillis());
			AntennaToPad a2p = new AntennaToPad(antennaToAdd, assignedPad, currentTime, null, false);
			BaseElementConversationUtils.getInstance().saveOrUpdatePad(a2p.getPad(), ConversationToken.CONVERSATION_COMPLETED);

			// notify any interested parties
			ConfigurationsView view = (ConfigurationsView)RcpUtils.findView(ConfigurationsView.ID);
			view.externalModelChange();
			
			// extra refresh to make sure we don't have an expanded tree node that has no children (due to deletion/unassignment)
			view.getConfigurationsTreeViewer().refresh();
		}
		catch( Exception e ) {
			throw new RuntimeException("Problem assigning Antenna to Pad.", e);
		}
	}
	
	public static void closeOutAntToPad(AntennaToPad toBeClosed, ConversationToken token) throws Exception 
	{
		if(null != toBeClosed) {
			Long endTime = UTCUtility.utcJavaToOmg(System.currentTimeMillis());
			toBeClosed.setEndTime(endTime);
			AntennaToPadConversationUtils.getInstance().saveOrUpdateAntennaToPad(toBeClosed, token);
		}
	}

	@Override
	public void doConversational() 
	{
		Shell shell = _window.getShell();

		try {
			// set a wait cursor
			_window.getShell().setCursor(_window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			
			// Collect user input
			ChoosePadWizard wizard = new ChoosePadWizard( antenna.getConfiguration() );
			WizardDialog dialog = new WizardDialog( shell, wizard );
			int ret = dialog.open();
			if( ret == WizardDialog.CANCEL ) {
				return;
			}

			Pad chosenPad = wizard.getPad();
			
			// check to see if the pad already has an antenna assigned to it; 
			// if so, prompt the user with a dialog e.g. "are you sure..."
			AntennaToPad[] existingAntennaAssignments = null;
			if(chosenPad.getConfiguration().getId().equals(antenna.getConfiguration().getId()))
			{
				existingAntennaAssignments = BaseElementConversationUtils.getInstance().findOpenAntennaToPadAssignmentsForPad(chosenPad, antenna.getConfiguration());
			}
			
			if(null != existingAntennaAssignments && existingAntennaAssignments.length > 0) 
			{
				StringBuilder antennaNames = new StringBuilder("");
				int count = 0;
				for(AntennaToPad assignment : existingAntennaAssignments)
				{
					if(antenna.getId().equals(assignment.getAntenna().getId())) {
						MessageDialog.openInformation(_window.getShell(), "Nothing to do here...", "The selected antenna and pad are already associated. This assignment has no effect.");
						return;
					}
					antennaNames.append(assignment.getAntenna().getName());
					if(! (++count >= existingAntennaAssignments.length) ) {
						antennaNames.append(", ");
					}
				}
				
				boolean doIt = MessageDialog.openConfirm(_window.getShell(), 
						"Please confirm","Pad " + 
						chosenPad.getName() + 
						" is currently assigned to antenna " 
						+ antennaNames.toString()
						+ ". If you continue, in addition to creating the new assignment, the previous assignment will be closed.");

				if(!doIt) {
					return;
				}
				
				// close out any previous antennaToPad mappings in which this pad is used within our (i.e. the antenna's) configuration. 
				// [note that if the pad is used in a _different_ configuration, it's ok and those mappings should _not_ be deleted]
				// we will search first for mappings in the antenna's configuration, and then mappings in the pad's configuration;
				for(AntennaToPad iteratedA2p : existingAntennaAssignments) 
				{
					// if the configs are the same, but the antennas are different, we have an assignment to close out
					if(antenna.getConfiguration().getId().equals(iteratedA2p.getAntenna().getConfiguration().getId()) && 
							!iteratedA2p.getAntenna().getId().equals(antenna.getId())) 
					{
						closeOutAntToPad(iteratedA2p, ConversationToken.CONVERSATION_COMPLETED);
					}
				}
			}
			
			// Create and save an antennaToPad assignment with the given info
			AssignAntennaToPadAction.addAntennaToPad(antenna, chosenPad);
		}
		catch( Exception e ) {
			e.printStackTrace();
			throw new RuntimeException("Could not create AntennaToPad assignment", e);
		} finally {
			// remove the wait cursor
			_window.getShell().setCursor(null);
		}
	}

	@Override
	public void doPostConversational() {
		// noop as it was already done in the static method
	}

	@Override
	public void selectionChanged(IWorkbenchPart wbpart, ISelection newSelection) 
	{
		if( newSelection instanceof IStructuredSelection && GuiUtils.isGodUser()) 
		{
			selection = (IStructuredSelection)newSelection;
			setEnabled( selection.size() == 1 && 
			            selection.getFirstElement() instanceof Antenna );
			if(isEnabled())
			{
				this.antenna = (Antenna) selection.getFirstElement();
			}
		}
		else {
			setEnabled(false);
		}
	}
	
	@Override
	public void dispose() {
		window.getSelectionService().removeSelectionListener(this);
	}

}
