/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import java.util.Date;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.ActionFactory.IWorkbenchAction;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.BaseElementConversationUtils;
import alma.obops.tmcdbgui.views.ConfigurationsView;
import alma.obops.tmcdbgui.views.providers.helpers.config.WeatherStationList;
import alma.obops.tmcdbgui.wizards.AddWeatherStationWizard;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.WeatherStationController;

/**
 * Action to create a new weather station within a configuration.
 * @author sharring
 *
 */
public class AddWeatherStationAction extends Action implements ISelectionListener, IWorkbenchAction, IAssignNewWeatherStationAttributes
{
	private final static String ID = "add_weatherstation.action";
	private HwConfiguration configuration;
	private IWorkbenchWindow _window;
	private IStructuredSelection selection;
	private String weatherstationname;
	private Long commissionDate;
	
	/**
	 * Constructor
	 * @param configuration the configuration in which the new weather station will "live".
	 * @param view the TreeViewer associated with the action.
	 */
	public AddWeatherStationAction(IWorkbenchWindow window)
	{
		_window = window;
		setId(ID);
		setText("New weather station");
		setToolTipText("Adds a new weather station");
		setImageDescriptor( RcpUtils.getImageDescriptor( "icons/weatherstation-new.png" ));
		_window.getSelectionService().addSelectionListener(this);
	}

	@Override
	public void selectionChanged(IWorkbenchPart part, ISelection newSelection) 
	{
		if( newSelection instanceof IStructuredSelection  && GuiUtils.isGodUser()) 
		{
			selection = (IStructuredSelection)newSelection;
			if( selection.size() == 1 &&
					   (selection.getFirstElement() instanceof WeatherStationList || 
					    selection.getFirstElement() instanceof HwConfiguration) )	
			{
				this.configuration = getConfigurationFromSelection();
				if(!configurationHasWeatherStationController())
				{
					setEnabled(true);
				}
				else 
				{
					setEnabled(false);
				}
			}
			else {
				setEnabled(false);
			}
		}
		else {
			setEnabled(false);
		}
	}

	private boolean configurationHasWeatherStationController() 
	{
		boolean retVal = false;
		for(BaseElement be : configuration.getBaseElements())
		{
			if(be.getType().equals(BaseElementType.WeatherStationController))
			{
				retVal = true;
			}
		}
		return retVal;
	}

	@Override
	public void dispose() 
	{
	}
	
	private HwConfiguration getConfigurationFromSelection() 
	{
		HwConfiguration conf = null;
		if( selection.getFirstElement() instanceof HwConfiguration ) {
			conf = (HwConfiguration)selection.getFirstElement();
		}
		else if( selection.getFirstElement() instanceof WeatherStationList ) {
			conf = ((WeatherStationList)selection.getFirstElement()).getHwConfiguration();
		}
		return conf;
	}
	
	@Override
	public final void run() 
	{
		Shell shell = _window.getShell();

		try 
		{
			// Collect user input
			_window.getShell().setCursor(_window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			AddWeatherStationWizard wizard = new AddWeatherStationWizard( this, configuration );
			WizardDialog dialog = new WizardDialog( shell, wizard );
			int ret = dialog.open();
			if( ret == WizardDialog.CANCEL ) {
				return;
			}

			// Create and save an weather station with the given info
			configuration = getConfigurationFromSelection();
			WeatherStationController newWeatherStation = new WeatherStationController();
			newWeatherStation.setName(weatherstationname);
			newWeatherStation.setConfiguration(configuration);
			newWeatherStation.setCommissionDate(commissionDate);
			newWeatherStation.setType(BaseElementType.WeatherStationController);
			newWeatherStation.getConfiguration().addBaseElement(newWeatherStation);

			_window.getShell().setCursor(_window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			BaseElementConversationUtils.getInstance().saveOrUpdateWeatherStation(newWeatherStation);
			ConfigurationsView configsView = (ConfigurationsView)RcpUtils.findView(ConfigurationsView.ID);
			configsView.internalModelChange();
		}
		catch( Exception e ) {
			throw new RuntimeException("Could not create weather station", e);
		} finally {
			_window.getShell().setCursor(null);
		}
}

	@Override
	public void setCommissionDate(Date commissionDate) {
		this.commissionDate = commissionDate.getTime();
	}

	@Override
	public void setWeatherStationName(String name) {
		this.weatherstationname = name;
	}
}
