/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import java.util.Date;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.obops.tmcdbgui.views.providers.typedlists.PadList;
import alma.obops.tmcdbgui.wizards.AddPadWizard;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.Coordinate;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.Pad;

/**
 * Action for adding a pad to a configuration.
 * @author sharring
 */
public class AddPadAction extends ConversationalAction 
{

	private String name;
	private Date commissionDate;
	private Coordinate position;
	private Double cableDelay;
	private boolean cancelled = false;
    private String ID = "add_pad.action";

	/**
	 * Constructor
	 * @param configuration the configuration in which the new antenna will "live".
	 * @param view the TreeViewer associated with the action.
	 */
	public AddPadAction(IWorkbenchWindow wbwindow, IModelChangeListener listener) 
	{
		window = wbwindow;
		setId(ID);
		setText("New Pad");
		setToolTipText("Adds a new Pad");
		setImageDescriptor( RcpUtils.getImageDescriptor( "icons/pad-new.png" ));
		window.getSelectionService().addSelectionListener(this);
		this.addModelChangeListener(listener);
	}
	
	/**
	 * Setter for the position (x, y, z coordinates) of the new pad.
	 * @param position the coordinates (x, y, z) of the new pad.
	 */
	public void setPosition(Coordinate position) {
		this.position = position;
	}

	/**
	 * Setter for the commission date of the new pad.
	 * @param commissionDate the date that the new pad is commissioned.
	 */
	public void setCommissionDate(Date commissionDate) {
		this.commissionDate = commissionDate;
	}

	/**
	 * Setter for the name of the new pad.
	 * @param name the name of the new pad.
	 */
	public void setName(String name) {
		this.name = name;
	}
	
	/**
	 * Setter for the cable delay of the new pad.
	 * @param delay the cable delay of the new pad.
	 */
	public void setCableDelay(Double delay) {
		this.cableDelay = delay;
	}
	
	@Override
	public void doPreConversational()
	{
		// Collect user input
		cancelled = false;

		try {
			window.getShell().setCursor(window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			AddPadWizard wizard = new AddPadWizard( this, getConfigurationFromSelection() );
			WizardDialog dialog = new WizardDialog( window.getShell(), wizard );
			int ret = dialog.open();

			if( ret == WizardDialog.CANCEL ) {
				cancelled = true;
				return;
			}
		}
		finally {
			window.getShell().setCursor(null);
		}
	}

	@Override
	public void doConversational() 
	{
		if(cancelled) {
			return;
		}

		try {
			window.getShell().setCursor(window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			// Create and save a pad with the given info
			Pad newPad = new Pad();
			newPad.setName(name);
			newPad.setConfiguration( getConfigurationFromSelection() );
			newPad.setType(BaseElementType.Pad);
			newPad.setPosition(position); // required non-null...
			newPad.setCommissionDate(commissionDate.getTime()); // required non-null...
			newPad.setAvgDelay(cableDelay);
			newPad.getConfiguration().addBaseElement(newPad);

			HwConfigurationConversationUtils.getInstance().updateConfiguration(newPad.getConfiguration());
		}
		catch( Exception e ) {
			e.printStackTrace();
			MessageDialog.openError( window.getShell(), 
					"Creation of Pad",
					e.getClass().getSimpleName() + 
					": " + e.getMessage() );
		} finally {
			window.getShell().setCursor(null);
		}
	}

	@Override
	public void doPostConversational() {
		this.modelChanged();
	}

	@Override
	public void selectionChanged(IWorkbenchPart part, ISelection newSelection) {
		if( newSelection instanceof IStructuredSelection && GuiUtils.isGodUser()) {
			selection = (IStructuredSelection)newSelection;
			if( selection.size() == 1 &&
			   (selection.getFirstElement() instanceof PadList || 
			    selection.getFirstElement() instanceof HwConfiguration) ) {
				setEnabled(true);
			}
			else {
				setEnabled(false);
			}
		}
		else {
			setEnabled(false);
		}
	}

	private HwConfiguration getConfigurationFromSelection() {
		HwConfiguration conf = null;
		if( selection.getFirstElement() instanceof HwConfiguration ) {
			conf = (HwConfiguration)selection.getFirstElement();
		}
		else if( selection.getFirstElement() instanceof PadList ) {
			conf = ((PadList)selection.getFirstElement()).getHwConfiguration();
		}
		return conf;
	}
}
