/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import java.util.Date;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.ActionFactory.IWorkbenchAction;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.BaseElementConversationUtils;
import alma.obops.tmcdbgui.views.ConfigurationsView;
import alma.obops.tmcdbgui.views.providers.helpers.config.HolographyTowerList;
import alma.obops.tmcdbgui.wizards.AddHolographyTowerWizard;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.Coordinate;
import alma.tmcdb.domain.HolographyTower;
import alma.tmcdb.domain.HwConfiguration;

public class AddHolographyTowerAction extends Action implements
		ISelectionListener, IWorkbenchAction,
		IAssignNewHolographyTowerAttributes 
{
	private final static String ID = "add_holographytower.action";
	private HwConfiguration configuration;
	private IWorkbenchWindow _window;
	private IStructuredSelection selection;
	private String holographytowername;
	private Long commissionDate;
	private Coordinate position;
	
	/**
	 * Constructor
	 * @param configuration the configuration in which the new weather station will "live".
	 * @param view the TreeViewer associated with the action.
	 */
	public AddHolographyTowerAction(IWorkbenchWindow window)
	{
		_window = window;
		setId(ID);
		setText("New holography tower");
		setToolTipText("Adds a new holography tower");
		setImageDescriptor( RcpUtils.getImageDescriptor( "icons/holographytower-new.png" ));
		_window.getSelectionService().addSelectionListener(this);
	}

	@Override
	public void selectionChanged(IWorkbenchPart part, ISelection newSelection) 
	{
		if( newSelection instanceof IStructuredSelection && GuiUtils.isGodUser()) 
		{
			selection = (IStructuredSelection)newSelection;
			if( selection.size() == 1 &&
					   (selection.getFirstElement() instanceof HolographyTowerList || 
					    selection.getFirstElement() instanceof HwConfiguration) )	
			{
				setEnabled(true);
			}
			else {
				setEnabled(false);
			}
		}
		else {
			setEnabled(false);
		}
	}

	@Override
	public void dispose() 
	{
	}
	
	private HwConfiguration getConfigurationFromSelection() 
	{
		HwConfiguration conf = null;
		if( selection.getFirstElement() instanceof HwConfiguration ) {
			conf = (HwConfiguration)selection.getFirstElement();
		}
		else if( selection.getFirstElement() instanceof HolographyTowerList ) {
			conf = ((HolographyTowerList)selection.getFirstElement()).getHwConfiguration();
		}
		return conf;
	}
	
	@Override
	public final void run() 
	{
		Shell shell = _window.getShell();

		try 
		{
			configuration = getConfigurationFromSelection();
			
			// Collect user input
			_window.getShell().setCursor(_window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			AddHolographyTowerWizard wizard = new AddHolographyTowerWizard( this, configuration );
			WizardDialog dialog = new WizardDialog( shell, wizard );
			int ret = dialog.open();
			if( ret == WizardDialog.CANCEL ) {
				return;
			}

			// Create and save a holography tower with the given info
			HolographyTower newHolographyTower = new HolographyTower();
			newHolographyTower.setName(holographytowername);
			newHolographyTower.setConfiguration(configuration);
			newHolographyTower.setCommissionDate(commissionDate);
			newHolographyTower.setPosition(position);
			newHolographyTower.setType(BaseElementType.HolographyTower);
			newHolographyTower.getConfiguration().addBaseElement(newHolographyTower);

			_window.getShell().setCursor(_window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			BaseElementConversationUtils.getInstance().saveOrUpdateHolographyTower(newHolographyTower);
			ConfigurationsView configsView = (ConfigurationsView)RcpUtils.findView(ConfigurationsView.ID);
			configsView.internalModelChange();
		}
		catch( Exception e ) {
			throw new RuntimeException("Could not create weather station", e);
		} finally {
			_window.getShell().setCursor(null);
		}
}

	@Override
	public void setCommissionDate(Date commissionDate) {
		this.commissionDate = commissionDate.getTime();
	}

	@Override
	public void setHolographyTowerName(String name) {
		this.holographytowername = name;
	}

	@Override
	public void setPosition(Coordinate position) {
		this.position = position;
	}

}
