/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import java.util.Date;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.BaseElementConversationUtils;
import alma.obops.tmcdbgui.views.providers.typedlists.FrontEndList;
import alma.obops.tmcdbgui.wizards.AddFrontendWizard;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.FrontEnd;
import alma.tmcdb.domain.HwConfiguration;

/**
 * Action to add a new frontend.
 * @author sharring
 */
public class AddFrontendAction extends ConversationalAction implements IAssignNewFrontendAttributes
{
	private String ID = "add_frontend.action";
	private String name;
	private IWorkbenchWindow _window;
	private HwConfiguration configuration;
	private Date commissionDate;
	
	/**
	 * Constructor
	 * @param configuration the configuration in which the new Frontend will "live".
	 * @param view the TreeViewer associated with the action.
	 */
	public AddFrontendAction(IWorkbenchWindow window, IModelChangeListener listener)
	{
		_window = window;
		setId(ID);
		setText("New frontend");
		setToolTipText("Adds a new Frontend");
		setImageDescriptor( RcpUtils.getImageDescriptor( "icons/front-end-new.png" ));
		_window.getSelectionService().addSelectionListener(this);
		this.addModelChangeListener(listener);
	}
	 
	@Override
	public void doConversational() 
	{
		Shell shell = _window.getShell();

		try {
			// Collect user input
			_window.getShell().setCursor(_window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			AddFrontendWizard wizard = new AddFrontendWizard( this, configuration );
			WizardDialog dialog = new WizardDialog( shell, wizard );
			int ret = dialog.open();
			if( ret == WizardDialog.CANCEL ) {
				return;
			}

			// Create and save an Frontend with the given info
			configuration = getConfigurationFromSelection();
			FrontEnd newFrontend = new FrontEnd();
			newFrontend.setId(null);
			newFrontend.setName(name);
			newFrontend.setConfiguration(configuration);
			newFrontend.setType(BaseElementType.FrontEnd);
			newFrontend.setCommissionDate(commissionDate.getTime()); // required non-null...
			newFrontend.getConfiguration().addBaseElement(newFrontend);

			BaseElementConversationUtils.getInstance().saveOrUpdateFrontEnd(newFrontend);
		}
		catch( Exception e ) {
			e.printStackTrace();
			throw new RuntimeException("Could not create Frontend", e);
		} finally {
			_window.getShell().setCursor(null);
		}
	}

	@Override
	public void doPostConversational() 
	{
		this.modelChanged();
	}

	@Override
	public void selectionChanged(IWorkbenchPart workbenchPart, ISelection newSelection) 
	{
		if( newSelection instanceof IStructuredSelection && GuiUtils.isGodUser()) 
		{
			selection = (IStructuredSelection)newSelection;
			if( selection.size() == 1 && 
					   (selection.getFirstElement() instanceof FrontEndList || 
					    selection.getFirstElement() instanceof HwConfiguration) )	
			{
				this.configuration = getConfigurationFromSelection();
				setEnabled(true);
			}
			else {
				setEnabled(false);
			}
		}
		else {
			setEnabled(false);
		}
	}
	
	private HwConfiguration getConfigurationFromSelection() 
	{
		HwConfiguration conf = null;
		if( selection.getFirstElement() instanceof HwConfiguration ) {
			conf = (HwConfiguration)selection.getFirstElement();
		}
		else if( selection.getFirstElement() instanceof FrontEndList ) {
			conf = ((FrontEndList)selection.getFirstElement()).getHwConfiguration();
		}
		return conf;
	}

	@Override
	public void setCommissionDate(Date commissionDate) {
		this.commissionDate = commissionDate;
	}

	@Override
	public void setName(String name) {
		this.name = name;
	}
}
