/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import java.lang.reflect.InvocationTargetException;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;

import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.DomainObjectUtils;
import alma.obops.tmcdbgui.utils.conversation.BaseElementConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.StartupScenarioConversationUtils;
import alma.obops.tmcdbgui.views.ConfigurationsView;
import alma.obops.tmcdbgui.wizards.AddBaseElementStartupsWizard;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementStartup;
import alma.tmcdb.domain.BaseElementStartupType;
import alma.tmcdb.domain.StartupScenario;

public class AddBaseElementsStartupAction extends ConversationalAction 
{
	private static final String ADD_BASE_ELEMENT_STARTUPS = "Add Base Element Startup(s)";
	private static final String ID = "add_baseelementstartup.action";
	private StartupScenario destinationStartupScenario;

	/**
	 * Constructor.
	 * @param window the window associated with this action.
	 */
	public AddBaseElementsStartupAction(IWorkbenchWindow window, IModelChangeListener listener) 
	{
		this.window = window;
		this.window.getSelectionService().addSelectionListener(this);
		this.listeners.add(listener);

		setImageDescriptor( RcpUtils.getImageDescriptor( "icons/baseelement-new.gif" ));
		setId(ID);
		setText( ADD_BASE_ELEMENT_STARTUPS );
		setToolTipText("Adds one or more new base element startups to a startup scenario");
	}

	public StartupScenario getDestinationStartupScenario() {
		return destinationStartupScenario;
	}

	public void setDestinationStartupScenario(
			StartupScenario destinationStartupScenario) {
		this.destinationStartupScenario = destinationStartupScenario;
	}

	@Override
	public void doConversational() 
	{
		final Shell shell = window.getShell();
		// Collect user input
		window.getShell().setCursor(window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
		AddBaseElementStartupsWizard wizard = new AddBaseElementStartupsWizard(destinationStartupScenario);
		wizard.init(window.getWorkbench(), selection);
		WizardDialog dialog = new WizardDialog( shell, wizard );
		dialog.create();
		int ret = dialog.open();
		if( ret == WizardDialog.CANCEL ) {
			return;
		}

		final BaseElement[] baseElements = wizard.getBaseElements();
		ProgressMonitorDialog pd = new ProgressMonitorDialog(window.getShell());
		try 
		{
			pd.run(true, true, new IRunnableWithProgress() 
			{
				public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException 
				{
					monitor.beginTask("Adding base element startup(s)", 1000 * baseElements.length);
					monitor.worked(1);

					BaseElementStartup[] baseElementStartups = createBaseElementStartupsForSelectedBaseElements(baseElements);
					for(BaseElementStartup bes : baseElementStartups) 
					{
						monitor.subTask("Processing base element: " + bes.getBaseElement().getName());
						try {
							monitor.worked(200);
							StartupScenarioConversationUtils.getInstance().saveOrUpdateBaseElementStartup(bes, ConversationToken.CONVERSATION_PENDING);
							monitor.worked(200);
							doNestedBaseElements(bes);
							monitor.worked(200);
							copySwItemsAsNeeded(bes);
							monitor.worked(400);
						} catch(Exception e) {
							MessageDialog.openError(shell, "Problem encountered", "Could not add startups to startup scenario");
							e.printStackTrace();
						}
					}
				}
			}
		);
		} catch(Exception e) {
			MessageDialog.openError(shell, "Problem encountered", "Could not add startups to startup scenario");
			e.printStackTrace();
		}
	}

	private void copySwItemsAsNeeded(BaseElementStartup bes) 
	{
		if(!bes.getBaseElement().getConfiguration().getId().equals(bes.getStartup().getConfiguration().getId()))
		{
			try {
				BaseElementConversationUtils.getInstance().copySwItemsForBaseElement(bes.getBaseElement(), destinationStartupScenario.getConfiguration());
			} catch (Exception e) {
				e.printStackTrace();
				throw new RuntimeException("Could not copy sw items for base element", e);
			}
		}
	}

	private void doNestedBaseElements(BaseElementStartup bes) throws Exception 
	{
		if(bes.getType().equals(BaseElementStartupType.Antenna)) 
		{
			// add a nested, generic frontend as well as the antenna
			BaseElementStartup frontEndStartup = new BaseElementStartup();
			frontEndStartup.setGeneric("true");
			frontEndStartup.setParent(bes);
			frontEndStartup.setSimulated(false);
			frontEndStartup.setType(BaseElementStartupType.FrontEnd);
			StartupScenarioConversationUtils.getInstance().saveOrUpdateBaseElementStartup(frontEndStartup, ConversationToken.CONVERSATION_PENDING);
		}
		else if(bes.getType().equals(BaseElementStartupType.CentralLO)) 
		{
			// add 6 nested, generic photonic refs as well as the CentralLO
			BaseElementStartup photonicRef1Startup = new BaseElementStartup();
			photonicRef1Startup.setGeneric("true");
			photonicRef1Startup.setParent(bes);
			photonicRef1Startup.setSimulated(false);
			photonicRef1Startup.setType(BaseElementStartupType.PhotonicReference1);
			StartupScenarioConversationUtils.getInstance().saveOrUpdateBaseElementStartup(photonicRef1Startup, ConversationToken.CONVERSATION_PENDING);
			
			BaseElementStartup photonicRef2Startup = new BaseElementStartup();
			photonicRef2Startup.setGeneric("true");
			photonicRef2Startup.setParent(bes);
			photonicRef2Startup.setSimulated(false);
			photonicRef2Startup.setType(BaseElementStartupType.PhotonicReference2);
			StartupScenarioConversationUtils.getInstance().saveOrUpdateBaseElementStartup(photonicRef2Startup, ConversationToken.CONVERSATION_PENDING);
			
			BaseElementStartup photonicRef3Startup = new BaseElementStartup();
			photonicRef3Startup.setGeneric("true");
			photonicRef3Startup.setParent(bes);
			photonicRef3Startup.setSimulated(false);
			photonicRef3Startup.setType(BaseElementStartupType.PhotonicReference3);
			StartupScenarioConversationUtils.getInstance().saveOrUpdateBaseElementStartup(photonicRef3Startup, ConversationToken.CONVERSATION_PENDING);
			
			BaseElementStartup photonicRef4Startup = new BaseElementStartup();
			photonicRef4Startup.setGeneric("true");
			photonicRef4Startup.setParent(bes);
			photonicRef4Startup.setSimulated(false);
			photonicRef4Startup.setType(BaseElementStartupType.PhotonicReference4);
			StartupScenarioConversationUtils.getInstance().saveOrUpdateBaseElementStartup(photonicRef4Startup, ConversationToken.CONVERSATION_PENDING);
			
			BaseElementStartup photonicRef5Startup = new BaseElementStartup();
			photonicRef5Startup.setGeneric("true");
			photonicRef5Startup.setParent(bes);
			photonicRef5Startup.setSimulated(false);
			photonicRef5Startup.setType(BaseElementStartupType.PhotonicReference5);
			StartupScenarioConversationUtils.getInstance().saveOrUpdateBaseElementStartup(photonicRef5Startup, ConversationToken.CONVERSATION_PENDING);
			
			BaseElementStartup photonicRef6Startup = new BaseElementStartup();
			photonicRef6Startup.setGeneric("true");
			photonicRef6Startup.setParent(bes);
			photonicRef6Startup.setSimulated(false);
			photonicRef6Startup.setType(BaseElementStartupType.PhotonicReference6);
			StartupScenarioConversationUtils.getInstance().saveOrUpdateBaseElementStartup(photonicRef6Startup, ConversationToken.CONVERSATION_PENDING);
		}
	}

	private BaseElementStartup[] createBaseElementStartupsForSelectedBaseElements(BaseElement[] baseElements)
	{
		BaseElementStartup[] retVal = new BaseElementStartup[baseElements.length];
		int i = 0;
		for(BaseElement be: baseElements)
		{
			BaseElementStartup bes = new BaseElementStartup();
			bes.setBaseElement(be);
			bes.setGeneric("false");
			bes.setId(null);
			bes.setParent(null);
			bes.setSimulated(false);
			bes.setType(DomainObjectUtils.getBaseElementStartupTypeFromBaseElementType(be.getType()));
			bes.setStartup(destinationStartupScenario);
			retVal[i++] = bes;
		}
		return retVal;
	}

	@Override
	public void doPostConversational() {
		Shell shell = window.getShell();
		try {
			shell.setCursor(shell.getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			ConfigurationsView view = (ConfigurationsView) RcpUtils.findView(ConfigurationsView.ID);
			view.externalModelChange();
//			view.internalModelChange();
		}
		finally {
			shell.setCursor(null);
		}
	}

	@Override
	public void selectionChanged(IWorkbenchPart workbench, ISelection theSelection) {
		setEnabled(true);
	}

}
