/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.hibernate.exception.ConstraintViolationException;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AssemblyConversationUtils;
import alma.obops.tmcdbgui.views.providers.helpers.config.ConfigHelperFactory;
import alma.obops.tmcdbgui.views.providers.helpers.config.ConfigurationHelper;
import alma.obops.tmcdbgui.views.providers.typedlists.AssemblyList;
import alma.obops.tmcdbgui.views.providers.typedlists.TypeList;
import alma.obops.tmcdbgui.wizards.AddAssemblyWizard;
import alma.tmcdb.domain.Assembly;
import alma.tmcdb.domain.AssemblyType;
import alma.tmcdb.domain.HwConfiguration;

/**
 * Action to add a new assembly to a configuration.
 * @author sharring
 */
public class AddAssemblyAction extends ConversationalAction implements IAssignNewAssemblyAttributes
{
	private String ID = "add_assembly.action";
	private HwConfiguration configuration;
	private String serialNumber;
	//private String data;
	private AssemblyType asstype;
	
	/**
	 * Constructor
	 * @param configuration the configuration in which the new Assembly will "live".
	 * @param view the TreeViewer associated with the action.
	 */
	public AddAssemblyAction(IWorkbenchWindow window, IModelChangeListener listener)
	{
		this.window = window;
		setId(ID);
		setText("New &Assembly");
		setToolTipText("Adds a new Assembly");
		setImageDescriptor( RcpUtils.getImageDescriptor( "icons/assembly-new.png" ));
		window.getSelectionService().addSelectionListener(this);
		this.addModelChangeListener(listener);
	}
	
	@Override
	public void doConversational() 
	{
		Shell shell = window.getShell();

		try {
			// set a wait cursor
			window.getShell().setCursor(window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			

			// HACK: use the configuration helper to hydrate the assemblies
			configuration = getConfigurationFromSelection();
			ConfigurationHelper helper = (ConfigurationHelper) ConfigHelperFactory.getHelper(configuration);
			helper.getChildren();
			// END HACK
			
			// Collect user input
			AddAssemblyWizard wizard = new AddAssemblyWizard( this, configuration, asstype );
			WizardDialog dialog = new WizardDialog( shell, wizard );
			int ret = dialog.open();
			if( ret == WizardDialog.CANCEL ) {
				return;
			}

			// Create and save an Assembly with the given info
			Assembly newAssembly = new Assembly();
			newAssembly.setId(null);
			newAssembly.setSerialNumber(serialNumber);
			newAssembly.setConfiguration(configuration);
			newAssembly.setAssemblyType(asstype);

			// save the new assembly
			AssemblyConversationUtils.getInstance().saveOrUpdateAssembly(newAssembly);
			
			configuration.addAssembly(newAssembly);
		}
		catch(ConstraintViolationException ex) 
		{
			MessageDialog.openWarning(window.getShell(), "Cannot add assembly", "Assembly cannot be added due to database constraints. Serial number not unique?" );
		}
		catch (Exception e) 
		{
			MessageDialog.openError(window.getShell(), "Cannot add assembly", "Assembly cannot be added for unknown reasons." );
			e.printStackTrace();
		} finally {
			// remove the wait cursor
			window.getShell().setCursor(null);
		}
	}

	@Override
	public void doPostConversational() 
	{
		this.modelChanged();
	}

	@Override
	public void selectionChanged(IWorkbenchPart part, ISelection newSelection) 
	{
		if( newSelection instanceof IStructuredSelection && GuiUtils.isGodUser()) 
		{
			selection = (IStructuredSelection)newSelection;
			if( selection.size() == 1 &&
					   (selection.getFirstElement() instanceof TypeList || 
					    selection.getFirstElement() instanceof HwConfiguration) )	
			{
				getConfigurationFromSelection();
				asstype = null;
				setEnabled(true);
			}
			else if(selection.size() == 1 && 
					(selection.getFirstElement() instanceof AssemblyList))
			{
				getConfigurationFromSelection();
				asstype = ((AssemblyList) selection.getFirstElement()).getAssemblyType();
				setEnabled(true);
			} 
			else {
				setEnabled(false);
			}
		}
		else {
			setEnabled(false);
		}
	}
	
	private HwConfiguration getConfigurationFromSelection() 
	{
		HwConfiguration conf = null;
		if( selection.getFirstElement() instanceof HwConfiguration ) {
			conf = (HwConfiguration)selection.getFirstElement();
		}
		else if( selection.getFirstElement() instanceof AssemblyList ) {
			conf = ((AssemblyList)selection.getFirstElement()).getHwConfiguration();
		} else if( selection.getFirstElement() instanceof TypeList) {
			conf = ((TypeList)selection.getFirstElement()).getHwConfiguration();
		}
		return conf;
	}

	@Override
	public void setAssemblyType(AssemblyType type) {
		this.asstype = type;
	}

	@Override
	public void setSerialNumber(String serNum) {
		this.serialNumber = serNum;
	}

}
