/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import java.util.Date;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.BaseElementConversationUtils;
import alma.obops.tmcdbgui.views.providers.typedlists.AntennaList;
import alma.obops.tmcdbgui.wizards.AddAntennaWizard;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.AntennaType;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.Coordinate;
import alma.tmcdb.domain.HwConfiguration;

/**
 * Action to add a new antenna to a configuration.
 * @author sharring
 */
public class AddAntennaAction extends ConversationalAction implements IAssignNewAntennaAttributes
{
	private String name;
	private AntennaType type;
	private Coordinate position, offset;
	private Double diameter;
	private Date commissionDate;
	private Double cableDelay;
	private Integer loOffsetting;
	private Integer walshSequence;
	private Integer correlatorInputBaseline;
	private Integer correlatorInputAca;
	private HwConfiguration configuration;

    private IWorkbenchWindow _window;

    private static final String ID = "add_antenna.action";

	/**
	 * Constructor
	 * @param configuration the configuration in which the new antenna will "live".
	 * @param view the TreeViewer associated with the action.
	 */
	public AddAntennaAction(IWorkbenchWindow window, IModelChangeListener listener)
	{
		_window = window;
		setId(ID);
		setText("New Antenna");
		setToolTipText("Adds a new Antenna");
		setImageDescriptor( RcpUtils.getImageDescriptor( "icons/antenna-new.png" ));
		_window.getSelectionService().addSelectionListener(this);
		this.addModelChangeListener(listener);
	}
	
	@Override
	public void doConversational() 
	{
		Shell shell = _window.getShell();

		try {
			// Collect user input
			_window.getShell().setCursor(_window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			AddAntennaWizard wizard = new AddAntennaWizard( this, configuration );
			WizardDialog dialog = new WizardDialog( shell, wizard );
			int ret = dialog.open();
			if( ret == WizardDialog.CANCEL ) {
				return;
			}

			// Create and save an antenna with the given info
			configuration = getConfigurationFromSelection();
			Antenna newAntenna = new Antenna();
			newAntenna.setName(name);
			newAntenna.setId(null);
			newAntenna.setConfiguration(configuration);
			newAntenna.setType(BaseElementType.Antenna);
			newAntenna.setAntennaType(type); // required non-null...
			newAntenna.setPosition(position); // required non-null...
			newAntenna.setOffset(offset); // required non-null...
			newAntenna.setDiameter(diameter); // required non-null...
			newAntenna.setCommissionDate(commissionDate.getTime()); // required non-null...
			newAntenna.setAvgDelay(cableDelay);
			
			newAntenna.setLoOffsettingIndex(loOffsetting);
			newAntenna.setWalshSeq(walshSequence);
			newAntenna.setCaiAca(correlatorInputAca);
			newAntenna.setCaiBaseline(this.correlatorInputBaseline);
			
			newAntenna.getConfiguration().addBaseElement(newAntenna);

			_window.getShell().setCursor(_window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			BaseElementConversationUtils.getInstance().saveOrUpdateAntenna(newAntenna);
		}
		catch( Exception e ) {
			throw new RuntimeException("Could not create antenna", e);
		} finally {
			_window.getShell().setCursor(null);
		}
	}

	public void doPostConversational() {
		this.modelChanged();
	}

	/**
	 * @return the name
	 */
	public String getName() {
		return name;
	}	

	/**
	 * Setter for the name of the antenna. 
	 * @param name the name of the antenna.
	 */
	public void setName( String name ) {
		this.name = name;
	}

	/**
	 * Setter for the antenna's type.
	 * @param antennaType the type of the antenna.
	 */
	public void setType(AntennaType antennaType) {
		this.type = antennaType;
	}
	
	/**
	 * Setter for the antenna's position.
	 * @param position the position of the antenna.
	 */
	public void setPosition(Coordinate position) {
		this.position = position;
	}
	
	/**
	 * Setter for the antenna's offset (x offset; y, z are always zero).
	 * @param offset the antenna's offset in x; y, z are always zero.
	 */
	public void setOffset(Coordinate offset) {
		this.offset = offset;
	}
	
	/**
	 * Setter for the diameter of the antenna. 
	 * @param diameter the antenna's diameter.
	 */
	public void setDiameter(Double diameter) {
		this.diameter = diameter;
	}
	
	/**
	 * Setter for the commission date of the antenna.
	 * @param date the antenna's commission date.
	 */
	public void setCommissionDate(Date date) {
		this.commissionDate = date;
	}

	/**
	 * Setter for the cable delay.
	 * @param cableDelay the cable delay for the antenna.
	 */
	public void setCableDelay(Double cableDelay) {
		this.cableDelay = cableDelay;
	}

	@Override
	public void selectionChanged(IWorkbenchPart part, ISelection newSelection) 
	{
		if( newSelection instanceof IStructuredSelection ) 
		{
			selection = (IStructuredSelection)newSelection;
			if( selection.size() == 1 && GuiUtils.isGodUser() &&
					   (selection.getFirstElement() instanceof AntennaList || 
					    selection.getFirstElement() instanceof HwConfiguration) )	
			{
				setConfiguration(getConfigurationFromSelection());
				setEnabled(true);
			}
			else {
				setEnabled(false);
			}
		}
		else {
			setEnabled(false);
		}
	}

	@Override
	public void dispose() {
		_window.getSelectionService().removeSelectionListener(this);
	}

	@Override
	public void setConfiguration(HwConfiguration config) {
		this.configuration = config;
	}
	
	private HwConfiguration getConfigurationFromSelection() {
		HwConfiguration conf = null;
		if( selection.getFirstElement() instanceof HwConfiguration ) {
			conf = (HwConfiguration)selection.getFirstElement();
		}
		else if( selection.getFirstElement() instanceof AntennaList ) {
			conf = ((AntennaList)selection.getFirstElement()).getHwConfiguration();
		}
		return conf;
	}

	@Override
	public void setLoOffsetting(Integer loOffsetting) {
		this.loOffsetting = loOffsetting;
	}

	@Override
	public void setWalshSequence(Integer walshSequence) {
		this.walshSequence = walshSequence;	
	}

	@Override
	public void setCorrelatorInputBaseline(Integer input) {
		correlatorInputBaseline = input;
	}
	
	@Override
	public void setCorrelatorInputAca(Integer input) {
		correlatorInputAca = input;
	}
}
