/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.ActionFactory.IWorkbenchAction;

import alma.acs.tmcdb.Computer;
import alma.acs.tmcdb.Container;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.LabelHelper;
import alma.obops.tmcdbgui.utils.conversation.ContainerConversationUtils;
import alma.obops.tmcdbgui.views.SoftwareDeploymentView;

/**
 * Abstract class for deploying/undeploying a container. In general, both actions refer
 * to setting the container's computer. Common code, then, is under this class
 *
 * @author rtobar, May 9th, 2011
 *
 */
public abstract class AbstractMoveContainerAction extends Action implements ISelectionListener, IWorkbenchAction {

	private IStructuredSelection _selection;

	protected List<Container> _selectedContainers = new ArrayList<Container>();
	protected IWorkbenchWindow _window;

	public AbstractMoveContainerAction(IWorkbenchWindow window) {
    	_window = window;
    	_window.getSelectionService().addSelectionListener(this);
	}

	@Override
	public void run() 
	{

		Container c = null;
		Computer targetComputer = null;
		try {
			targetComputer = getTargetComputer();
		} catch (NoComputerSelectedException e1) { 
			return;
		}

		for(int i=0; i!=_selectedContainers.size(); i++)
			_selectedContainers.get(i).setComputer(targetComputer);

		try {
			for(int i=0; i!=_selectedContainers.size(); i++) {
				c = _selectedContainers.get(i);
				ContainerConversationUtils.getInstance().saveOrUpdateContainer(c);
			}
		} catch(Exception e) {
			e.printStackTrace();
			MessageDialog.openError(_window.getShell(),
			    "Error while un-assigning container",
			    "There was an unexpected error while un-assigning container '" + LabelHelper.getFullPath(c,false) + "'");
		}

		SoftwareDeploymentView view = (SoftwareDeploymentView) RcpUtils.findView(SoftwareDeploymentView.ID);
		view.internalModelChange();
	}
	
	@Override
	public void dispose() {
		_window.getSelectionService().removeSelectionListener(this);
	}

	@Override
	public void selectionChanged(IWorkbenchPart part, ISelection selection) {
		if( selection instanceof IStructuredSelection && GuiUtils.isGodUser()) 
		{
			_selectedContainers.clear();
			_selection = (IStructuredSelection)selection;
			if( selectionOnlyOfClass(_selection, Container.class, _selectedContainers) )
				setEnabled(true);
			else
				setEnabled(false);
		}
		else {
			setEnabled(false);
		}
	}

	@SuppressWarnings("unchecked")
	private <T> boolean selectionOnlyOfClass(IStructuredSelection selection, Class<T> clazz, List<T> objects) {

		if( selection.size() == 0 )
			return false;

		for (Iterator<?> it = selection.iterator(); it.hasNext();) {
			Object item = it.next();
			if( !item.getClass().equals(clazz) )
				return false;
			if( !validateSelectionItem(item) )
				return false;
			objects.add((T)item);
		}
		return true;
	}

	protected abstract boolean validateSelectionItem(Object item);
	protected abstract Computer getTargetComputer() throws NoComputerSelectedException;

	protected static class NoComputerSelectedException extends Exception {
		private static final long serialVersionUID = -4951804592453088749L;
	}

}
