/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.handlers;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.hibernate.exception.ConstraintViolationException;

import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.conversation.BackendConversationUtils;
import alma.obops.tmcdbgui.views.ConfigurationsView;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementType;

/**
 * Common abstract base class for all actions which need to delete a base element.
 * Child classes must implement the 2 abstract methods.
 * 
 * @author sharring
 */
public abstract class AbstractDeleteBaseElementAction extends ConversationalAction 
{
	private IStructuredSelection selection;
	private BaseElement baseElement;
	protected boolean cancelled;
	
	public AbstractDeleteBaseElementAction(IWorkbenchWindow window)
	{
		this.window = window;
		this.window.getSelectionService().addSelectionListener(this);
	}
	
	@Override public void doPreConversational() 
	{
		cancelled = !(MessageDialog.openConfirm(this.window.getShell(), "Please confirm", "Delete base element: '" + baseElement.getName() + "'?"));
	}
	
	@Override
	public void doConversational() 
	{
		if(cancelled) {
			return;
		}
		try {
			BackendConversationUtils.getInstance().delete(baseElement, ConversationToken.CONVERSATION_COMPLETED);
		} 
		catch(ConstraintViolationException ex) 
		{
			MessageDialog.openWarning(this.window.getShell(), "Cannot delete baseelement", getConstraintViolationErrorMessage() );
		}
		catch (Exception e) 
		{
			MessageDialog.openError(this.window.getShell(), "Cannot delete baseelement", "This baseelement cannot be deleted for unknown reasons." );
			e.printStackTrace();
		}
	}

	@Override
	public void doPostConversational() {
		if(cancelled) {
			return;
		}
		ConfigurationsView view = (ConfigurationsView) RcpUtils.findView(ConfigurationsView.ID);
		view.externalModelChange();
	}

	@Override
	public void selectionChanged(IWorkbenchPart workbenchPart, ISelection newSelection) 
	{
		if( newSelection instanceof IStructuredSelection ) 
		{
			selection = (IStructuredSelection)newSelection;
			if( selection.size() == 1 && (selection.getFirstElement() instanceof BaseElement))
			{
				this.baseElement = (BaseElement) selection.getFirstElement();
				if(this.baseElement.getType().equals(getType()))
				{
					setEnabled(true);
				} 
				else 
				{
					setEnabled(false);
				}
			}
			else 
			{
				setEnabled(false);
			}
		}
		else {
			setEnabled(false);
		}
	}
	
	/**
	 * In the event that the base element cannot be deleted due to a constraint violation, 
	 * this string will be presented to the user in an information dialog.
	 * 
	 * @return the string to use when the base element cannot be deleted due to db constraints.
	 */
	protected abstract String getConstraintViolationErrorMessage();
	
	/**
	 * The type of base element to delete.
	 * @return the base element's type.
	 */
	protected abstract BaseElementType getType();
}
