/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/

/*
 * Created on Oct 11, 2004
 */
package alma.obops.tmcdbgui.external.xmleditor.markers;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Logger;

import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.ITypedRegion;
import org.eclipse.ui.texteditor.MarkerUtilities;
import org.xml.sax.SAXParseException;

import alma.obops.tmcdbgui.external.xmleditor.xml.XMLValidationError;
import alma.obops.tmcdbgui.external.xmleditor.xml.XMLValidationErrorHandler;


/**
 * @author Phil Zoio
 */
public class MarkingErrorHandler extends XMLValidationErrorHandler
{

	private static final String ERROR_MARKER_ID = "TmcdbExplorer.xmlerror";
	private IResource resource;
	private IDocument document;
	private String    objectName;

	public MarkingErrorHandler(IResource resource, IDocument document, String objectName, Logger logger)
	{
		super(logger);
		this.resource   = resource;
		this.document   = document;
		this.objectName = objectName;
	}

	public void removeExistingMarkers()
	{
		try
		{
			resource.deleteMarkers(ERROR_MARKER_ID, true, IResource.DEPTH_ZERO);
		}
		catch (CoreException e1)
		{
			e1.printStackTrace();
		}
	}

	protected XMLValidationError nextError(SAXParseException e, boolean isFatal)
	{

		XMLValidationError validationError = super.nextError(e, isFatal);

		try
		{
			int lineNumber = e.getLineNumber();
			int columnNumber = e.getColumnNumber();

			Map<Object, Object> map = new HashMap<Object, Object>();
			map.put(IMarker.SEVERITY, Integer.valueOf(IMarker.SEVERITY_ERROR));
			map.put(IMarker.LOCATION, objectName);
			map.put(IMarker.LINE_NUMBER, lineNumber);
			map.put(IMarker.MESSAGE, e.getMessage());

			Integer charStart = getCharStart(lineNumber, columnNumber);
			if (charStart != null)
				map.put(IMarker.CHAR_START, charStart);

			Integer charEnd = getCharEnd(lineNumber, columnNumber);
			if (charEnd != null)
				map.put(IMarker.CHAR_END, charEnd);

			MarkerUtilities.createMarker(resource, map, ERROR_MARKER_ID);
		}
		catch (CoreException ee)
		{
			ee.printStackTrace();
		}

		return validationError;

	}

	private Integer getCharEnd(int lineNumber, int columnNumber)
	{
		try
		{
			return document.getLineOffset(lineNumber - 1) + columnNumber;
		}
		catch (BadLocationException e)
		{
			e.printStackTrace();
			return null;
		}
	}

	private Integer getCharStart(int lineNumber, int columnNumber)
	{
		try
		{
			int lineStartChar = document.getLineOffset(lineNumber - 1);
			Integer charEnd = getCharEnd(lineNumber, columnNumber);
			if (charEnd != null)
			{
				ITypedRegion typedRegion = document.getPartition(charEnd.intValue()-2);
				int partitionStartChar = typedRegion.getOffset();
				return partitionStartChar;
			}
			return lineStartChar;
		}
		catch (BadLocationException e)
		{
			return null;
		}
	}

}