/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.external.xmleditor;

import java.util.logging.Logger;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IPageLayout;
import org.eclipse.ui.editors.text.TextEditor;
import org.eclipse.ui.views.contentoutline.IContentOutlinePage;
import org.xml.sax.helpers.LocatorImpl;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.TmcdbGui;
import alma.obops.tmcdbgui.editors.inputs.IXmlEditorInput;
import alma.obops.tmcdbgui.external.xmleditor.markers.MarkingErrorHandler;
import alma.obops.tmcdbgui.external.xmleditor.outline.EditorContentOutlinePage;
import alma.obops.tmcdbgui.external.xmleditor.xml.XMLParser;


public class XMLEditor extends TextEditor
{

	public static String ID = "xml.editor";
	private Logger logger;
	private ColorManager colorManager;
	private IXmlEditorInput xmlEditorInput;
	private EditorContentOutlinePage outlinePage;
	private IResource workspaceRoot = ResourcesPlugin.getWorkspace().getRoot();
	private XMLDocumentProvider docProvider;

	public XMLEditor() {
		this(TmcdbGui.getLogger());
	}

	public XMLEditor(Logger logger)
	{
		super();
		this.logger = logger;
		colorManager = new ColorManager();
		setSourceViewerConfiguration(new XMLConfiguration(colorManager));
		docProvider = new XMLDocumentProvider();
		setDocumentProvider(docProvider);
	}

	public void dispose()
	{
		colorManager.dispose();
		if (outlinePage != null)
			outlinePage.setInput(null);
		super.dispose();
	}

	protected void doSetInput(IEditorInput newInput) throws CoreException
	{
		super.doSetInput(newInput);
		
		// we're overriding the inherited method, so this cast is, unfortunately, 
		// necessary (we can't change the signature of the overridden method).
		this.xmlEditorInput = (IXmlEditorInput)newInput;
		
		// configure the document provider in readonly (or read-write) mode, as appropriate
		docProvider.setReadOnly(xmlEditorInput.isReadOnly());

		if (outlinePage != null)
			outlinePage.setInput(xmlEditorInput);
		
		validateAndMark();
	}
	
	protected void editorSaved()
	{

		if(validateAndMark()) {
			super.editorSaved();
			xmlEditorInput.updateConfiguration();
		}
		// TODO: else, don't let us save the contents of the file

		if (outlinePage != null)
			outlinePage.update();				
	}

	protected boolean validateAndMark()
	{
		boolean retVal = false;
		try
		{
			IDocument document = getInputDocument();
			String text = document.get();
			MarkingErrorHandler markingErrorHandler = new MarkingErrorHandler(workspaceRoot, document, xmlEditorInput.getName(), logger);
			markingErrorHandler.setDocumentLocator(new LocatorImpl());
			markingErrorHandler.removeExistingMarkers();
			RcpUtils.findView(IPageLayout.ID_PROBLEM_VIEW, false);

			XMLParser parser = new XMLParser();
			parser.setErrorHandler(markingErrorHandler);
			parser.doParse(text, xmlEditorInput.getXmlSchemaForEditedObject());
			
			retVal = true;
		}
		catch (Exception e) { }

		return retVal;
	}

	protected IDocument getInputDocument()
	{
		IDocument document = getDocumentProvider().getDocument(xmlEditorInput);
		return document;
	}
	
	public IEditorInput getInput()
	{
		return xmlEditorInput;
	}

	/**
	 * Needed for content assistant
	 */
	protected void createActions()
	{
		super.createActions();
//		ResourceBundle bundle = Platform.getResourceBundle(TmcdbExplorer.getDefault().getBundle());
//		setAction("ContentFormatProposal", new TextOperationAction(bundle, "ContentFormatProposal.", this,
//				ISourceViewer.FORMAT));
//		setAction("ContentAssistProposal", new TextOperationAction(bundle, "ContentAssistProposal.", this,
//				ISourceViewer.CONTENTASSIST_PROPOSALS));
//		setAction("ContentAssistTip", new TextOperationAction(bundle, "ContentAssistTip.", this,
//				ISourceViewer.CONTENTASSIST_CONTEXT_INFORMATION));

	}
	
	
	@SuppressWarnings("unchecked")
	public Object getAdapter(Class required)
	{

		if (IContentOutlinePage.class.equals(required))
		{
			if (outlinePage == null)
			{
				outlinePage = new EditorContentOutlinePage(this);
				if (getEditorInput() != null)
					outlinePage.setInput(getEditorInput());
			}
			return outlinePage;
		}

		return super.getAdapter(required);
		
	}

}