/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.editors.inputs;

import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IPersistableElement;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.tmcdb.domain.PointingModel;
import alma.tmcdb.history.HistoryRecord;

/**
 * EditorInput for historical pointing model editor.
 * @author sharring
 *
 */
public class HistoricalPointingModelEditorInput implements IEditorInput 
{
	private PointingModel referencePm;
	private PointingModel previousPm;
	private HistoryRecord historyRecord;
	private String overrideName;
	
	public HistoricalPointingModelEditorInput(PointingModel referencePm, PointingModel previousPm, HistoryRecord record) 
	{
		this.referencePm = referencePm;
		this.previousPm = previousPm;
		this.historyRecord = record;
	}
	
	public HistoricalPointingModelEditorInput(PointingModel previousPm, PointingModel referencePm, HistoryRecord record, String name) 
	{
		this(previousPm, referencePm, record);
		this.overrideName = name;
	}

	@Override
	public boolean exists() {
		return false;
	}

	@Override
	public ImageDescriptor getImageDescriptor() {
		return RcpUtils.getImageDescriptor("icons/pointingmodel-history.png");
	}

	public String getName() 
	{
		String retVal = null;
		if(null != overrideName)
		{
			retVal = overrideName;
		}
		else 
		{
			retVal = "Pointing model v." + historyRecord.getVersion().toString() + " for " + referencePm.getAntenna().getName();			
		}
		return retVal;
	}

	@Override
	public IPersistableElement getPersistable() {
		return null;
	}

	@Override
	public String getToolTipText() {
		return getName() + " " + referencePm.getAntenna().getConfiguration().getName();
	}

	@SuppressWarnings("unchecked")
	@Override
	public Object getAdapter(Class adapter) {
		return null;
	}
	
	public PointingModel getReferencePointingModel()
	{
		return this.referencePm;
	}
	
	public PointingModel getPreviousPointingModel()
	{
		return this.previousPm;
	}
	
	public HistoryRecord getHistoryRecord() { return historyRecord; }
	
	public boolean equals(Object o) 
	{
		if(super.equals(o))
			return true;
		if( !(o instanceof HistoricalPointingModelEditorInput) )
			return false;

		HistoricalPointingModelEditorInput pmEditorInput = (HistoricalPointingModelEditorInput)o;
		PointingModel inputPm = pmEditorInput.getReferencePointingModel();

		if(inputPm.getId() == null && this.getReferencePointingModel().getId() != null) {
			return false;
		}
		else if(inputPm.getId() != null && this.getReferencePointingModel().getId() == null) {
			return false;
		}
		else if(inputPm.getId() == null && this.getReferencePointingModel().getId() == null) 
		{
		   if(inputPm.getAntenna().equals(this.getReferencePointingModel().getAntenna()) && 
			  pmEditorInput.getHistoryRecord().getVersion().equals(this.getHistoryRecord().getVersion()))
		   {
			   return true;
		   }
		return false;
		}
		
		if( inputPm.getId() != null && this.getReferencePointingModel().getId() != null ) 
		{
			if( this.getReferencePointingModel().getId().equals(inputPm.getId()) &&
				this.getHistoryRecord().getVersion().equals(pmEditorInput.getHistoryRecord().getVersion()) )
				return true;
			return false;
		}

		return false;
	}

	@Override
	public int hashCode() {
		int retVal = 17;
		retVal += this.getHistoryRecord() == null ? 0 : this.getHistoryRecord().getVersion().hashCode();
		return retVal;
	}
}
