/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.editors.inputs;

import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IPersistableElement;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.tmcdb.domain.FocusModel;
import alma.tmcdb.history.HistoryRecord;

/**
 * EditorInput for historical focus model editor.
 * @author sharring
 *
 */
public class HistoricalFocusModelEditorInput implements IEditorInput 
{
	private FocusModel referenceFocusModel;
	private FocusModel previousFocusModel;
	private HistoryRecord historyRecord;
	private String overrideName;
	
	public HistoricalFocusModelEditorInput(FocusModel pm, FocusModel previousPm, HistoryRecord record) 
	{
		this.referenceFocusModel = pm;
		this.previousFocusModel = previousPm;
		this.historyRecord = record;
	}

	public HistoricalFocusModelEditorInput(FocusModel pm, FocusModel previousPm, HistoryRecord record, String name) 
	{
		this(pm, previousPm, record);
		this.overrideName = name;
	}
	
	public FocusModel getReferenceFocusModel() {
		return this.referenceFocusModel;
	}
	
	public FocusModel getPreviousFocusModel() {
		return this.previousFocusModel;
	}
	
	@Override
	public boolean exists() {
		return false;
	}

	@Override
	public ImageDescriptor getImageDescriptor() {
		return RcpUtils.getImageDescriptor("icons/focusmodel-history.png");
	}

	public String getName() 
	{
		String retVal = null;
		if(null != overrideName) {
			retVal = overrideName;
		}
		else {
			retVal = "Focus model v." + historyRecord.getVersion().toString() + " for " + referenceFocusModel.getAntenna().getName();	
		}
		return retVal;
	}

	@Override
	public IPersistableElement getPersistable() {
		return null;
	}

	@Override
	public String getToolTipText() {
		return getName() + " in " + referenceFocusModel.getAntenna().getConfiguration().getName();
	}

	@SuppressWarnings("unchecked")
	@Override
	public Object getAdapter(Class adapter) {
		return null;
	}
	
	public FocusModel getCurrentFocusModel()
	{
		return this.previousFocusModel;
	}
	
	public HistoryRecord getHistoryRecord() { return historyRecord; }
	
	public boolean equals(Object o) 
	{
		if(super.equals(o))
			return true;
		if( !(o instanceof HistoricalFocusModelEditorInput) )
			return false;

		HistoricalFocusModelEditorInput fmEditorInput = (HistoricalFocusModelEditorInput)o;
		FocusModel inputPm = fmEditorInput.getReferenceFocusModel();

		if(inputPm.getId() == null && this.getReferenceFocusModel().getId() != null) {
			return false;
		}
		else if(inputPm.getId() != null && this.getReferenceFocusModel().getId() == null) {
			return false;
		}
		else if(inputPm.getId() == null && this.getReferenceFocusModel().getId() == null) 
		{
		   if(inputPm.getAntenna().equals(this.getReferenceFocusModel().getAntenna()) && 
			  fmEditorInput.getHistoryRecord().getVersion().equals(this.getHistoryRecord().getVersion()))
		   {
			   return true;
		   }
		return false;
		}
		
		if( inputPm.getId() != null && this.getReferenceFocusModel().getId() != null ) 
		{
			if( this.getReferenceFocusModel().getId().equals(inputPm.getId()) &&
				this.getHistoryRecord().getVersion().equals(fmEditorInput.getHistoryRecord().getVersion()) )
				return true;
			return false;
		}

		return false;
	}

	@Override
	public int hashCode() {
		int retVal = 17;
		retVal += this.getHistoryRecord() == null ? 0 : this.getHistoryRecord().getVersion().hashCode();
		return retVal;
	}
}
