/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.editors.inputs;

import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IPersistableElement;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.history.HistoryRecord;

/**
 * @author sharring
 *
 */
public class HistoricalAntennaEditorInput implements IEditorInput 
{

	private Antenna referenceAntenna;
	private Antenna previousAntenna;
	private HistoryRecord historyRecord;
	private String overrideName;
	
	public HistoricalAntennaEditorInput(Antenna antenna, Antenna previousAntenna, HistoryRecord record) 
	{
		this.referenceAntenna = antenna;
		this.previousAntenna = previousAntenna;
		this.historyRecord = record;
	}

	public HistoricalAntennaEditorInput(Antenna antenna, Antenna previousAntenna, HistoryRecord record, String name) 
	{
		this(antenna, previousAntenna, record);
		this.overrideName = name;
	}
	
	public Antenna getReferenceAntenna() {
		return this.referenceAntenna;
	}
	
	public Antenna getPreviousAntenna() {
		return this.previousAntenna;
	}
	
	@Override
	public boolean exists() {
		return false;
	}

	@Override
	public ImageDescriptor getImageDescriptor() {
		return RcpUtils.getImageDescriptor("icons/antenna-history.png");
	}

	public String getName() 
	{
		String retVal = null;
		if(null != overrideName) {
			retVal = overrideName;
		}
		else {
			retVal = "Antenna v." + historyRecord.getVersion().toString() + " for " + referenceAntenna.getName();	
		}
		return retVal;
	}

	@Override
	public IPersistableElement getPersistable() {
		return null;
	}

	@Override
	public String getToolTipText() {
		return getName();
	}

	@SuppressWarnings("unchecked")
	@Override
	public Object getAdapter(Class adapter) {
		return null;
	}
	
	public Antenna getCurrentAntenna()
	{
		return this.previousAntenna;
	}
	
	public HistoryRecord getHistoryRecord() { return historyRecord; }
	
	public boolean equals(Object o) 
	{
		if(super.equals(o))
			return true;
		if( !(o instanceof HistoricalAntennaEditorInput) )
			return false;

		HistoricalAntennaEditorInput antennaEditorInput = (HistoricalAntennaEditorInput)o;
		Antenna inputAntenna = antennaEditorInput.getReferenceAntenna();

		if(inputAntenna.getId() == null && this.getReferenceAntenna().getId() != null) {
			return false;
		}
		else if(inputAntenna.getId() != null && this.getReferenceAntenna().getId() == null) {
			return false;
		}
		else if(inputAntenna.getId() == null && this.getReferenceAntenna().getId() == null) 
		{
		   if(inputAntenna.equals(this.getReferenceAntenna()) && 
			  antennaEditorInput.getHistoryRecord().getVersion().equals(this.getHistoryRecord().getVersion()))
		   {
			   return true;
		   }
		return false;
		}
		
		if( inputAntenna.getId() != null && this.getReferenceAntenna().getId() != null ) 
		{
			if( this.getReferenceAntenna().getId().equals(inputAntenna.getId()) &&
				this.getHistoryRecord().getVersion().equals(antennaEditorInput.getHistoryRecord().getVersion()) )
				return true;
			return false;
		}

		return false;
	}

	@Override
	public int hashCode() {
		int retVal = 17;
		retVal += this.getHistoryRecord().getVersion() == null ? 0: this.getHistoryRecord().getVersion().hashCode();
		return retVal;
	}

}