/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.editors.inputs;

import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IPersistableElement;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.tmcdb.domain.AcaCorrDelays;
import alma.tmcdb.history.HistoryRecord;

/**
 * @author sharring
 *
 */
public class HistoricalAcaCorrDelaysEditorInput implements IEditorInput
{

	private AcaCorrDelays referenceAcaCorrDelays;
	private AcaCorrDelays previousAcaCorrDelays;
	private HistoryRecord historyRecord;
	private String overrideName;
	
	public HistoricalAcaCorrDelaysEditorInput(AcaCorrDelays delays, AcaCorrDelays previousAcaCorrDelays, HistoryRecord record) 
	{
		this.referenceAcaCorrDelays = delays;
		this.previousAcaCorrDelays = previousAcaCorrDelays;
		this.historyRecord = record;
	}

	public HistoricalAcaCorrDelaysEditorInput(AcaCorrDelays delays, AcaCorrDelays previousAcaCorrDelays, HistoryRecord record, String name) 
	{
		this(delays, previousAcaCorrDelays, record);
		this.overrideName = name;
	}
	
	public AcaCorrDelays getReferenceAcaCorrDelays() {
		return this.referenceAcaCorrDelays;
	}
	
	public AcaCorrDelays getPreviousAcaCorrDelays() {
		return this.previousAcaCorrDelays;
	}
	
	@Override
	public boolean exists() {
		return false;
	}

	@Override
	public ImageDescriptor getImageDescriptor() {
		return RcpUtils.getImageDescriptor("icons/delays.png");
	}

	public String getName() 
	{
		String retVal = null;
		if(null != overrideName) {
			retVal = overrideName;
		}
		else {
			String identifier = referenceAcaCorrDelays.getAntenna().getName();
			retVal = "AcaCorrDelays v." + historyRecord.getVersion().toString() + " for " + identifier;
		}
		return retVal;
	}

	@Override
	public IPersistableElement getPersistable() {
		return null;
	}

	@Override
	public String getToolTipText() {
		return getName();
	}

	@SuppressWarnings("unchecked")
	@Override
	public Object getAdapter(Class adapter) {
		return null;
	}
	
	public AcaCorrDelays getCurrentAcaCorrDelays()
	{
		return this.previousAcaCorrDelays;
	}
	
	public HistoryRecord getHistoryRecord() { return historyRecord; }
	
	public boolean equals(Object o) 
	{
		if(super.equals(o))
			return true;
		if( !(o instanceof HistoricalAcaCorrDelaysEditorInput) )
			return false;

		HistoricalAcaCorrDelaysEditorInput a2pEditorInput = (HistoricalAcaCorrDelaysEditorInput)o;
		AcaCorrDelays inputAcaCorrDelays = a2pEditorInput.getReferenceAcaCorrDelays();

		if(inputAcaCorrDelays.getAntennaId() == null && this.getReferenceAcaCorrDelays().getAntennaId() != null) {
			return false;
		}
		else if(inputAcaCorrDelays.getAntennaId() != null && this.getReferenceAcaCorrDelays().getAntennaId() == null) {
			return false;
		}
		else if(inputAcaCorrDelays.getAntennaId() == null && this.getReferenceAcaCorrDelays().getAntennaId() == null) 
		{
		   if(inputAcaCorrDelays.equals(this.getReferenceAcaCorrDelays()) && 
			  a2pEditorInput.getHistoryRecord().getVersion().equals(this.getHistoryRecord().getVersion()))
		   {
			   return true;
		   }
		   return false;
		}
		
		if( inputAcaCorrDelays.getAntennaId() != null && this.getReferenceAcaCorrDelays().getAntennaId() != null ) 
		{
			if( this.getReferenceAcaCorrDelays().getAntennaId().equals(inputAcaCorrDelays.getAntennaId()) &&
				this.getHistoryRecord().getVersion().equals(a2pEditorInput.getHistoryRecord().getVersion()) ) {
				return true;
			}
			return false;
		}

		return false;
	}

	@Override
	public int hashCode() {
		int retVal = 17;
		retVal += this.getHistoryRecord().getVersion() == null ? 0: this.getHistoryRecord().getVersion().hashCode();
		return retVal;
	}

}